<?php
/**
 * NEW FILE: cms-core/public/webview/admin/performance_settings.php
 * Advanced Performance Settings - Redis, Memcached, APCu, Object Storage
 */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';
require_once __DIR__ . '/../../../backend/performance/AdvancedCacheManager.php';

$errors = [];
$success = false;

// Handle form submissions
if (is_post()) {
    try {
        CSRF::require();
    } catch (Exception $e) {
        set_flash('error', 'Ungültiger CSRF-Token');
        redirect('/admin/performance_settings.php');
        exit;
    }
    
    $action = post('action');
    
    // Test Redis connection
    if ($action === 'test_redis') {
        $config = [
            'host' => trim(post('redis_host')),
            'port' => (int)post('redis_port'),
            'password' => post('redis_password'),
        ];
        
        if (AdvancedCacheManager::testConnection(AdvancedCacheManager::DRIVER_REDIS, $config)) {
            set_flash('success', '✅ Redis Verbindung erfolgreich!');
        } else {
            set_flash('error', '❌ Redis Verbindung fehlgeschlagen. Prüfe Host, Port und Passwort.');
        }
        
        redirect('/admin/performance_settings.php');
        exit;
    }
    
    // Save Redis settings
    if ($action === 'save_redis') {
        set_setting('redis_host', trim(post('redis_host')), 'text');
        set_setting('redis_port', (int)post('redis_port'), 'text');
        set_setting('redis_password', post('redis_password'), 'text');
        set_setting('redis_database', (int)post('redis_database', 0), 'text');
        
        $enabled = isset($_POST['redis_enabled']) && $_POST['redis_enabled'] === '1';
        set_setting('redis_enabled', $enabled, 'boolean');
        
        if ($enabled) {
            set_setting('cache_driver', 'redis', 'text');
        }
        
        set_flash('success', 'Redis-Einstellungen gespeichert.');
        redirect('/admin/performance_settings.php');
        exit;
    }
    
    // Test Memcached connection
    if ($action === 'test_memcached') {
        $config = [
            'host' => trim(post('memcached_host')),
            'port' => (int)post('memcached_port'),
        ];
        
        if (AdvancedCacheManager::testConnection(AdvancedCacheManager::DRIVER_MEMCACHED, $config)) {
            set_flash('success', '✅ Memcached Verbindung erfolgreich!');
        } else {
            set_flash('error', '❌ Memcached Verbindung fehlgeschlagen.');
        }
        
        redirect('/admin/performance_settings.php');
        exit;
    }
    
    // Save Memcached settings
    if ($action === 'save_memcached') {
        set_setting('memcached_host', trim(post('memcached_host')), 'text');
        set_setting('memcached_port', (int)post('memcached_port'), 'text');
        
        $enabled = isset($_POST['memcached_enabled']) && $_POST['memcached_enabled'] === '1';
        set_setting('memcached_enabled', $enabled, 'boolean');
        
        if ($enabled) {
            set_setting('cache_driver', 'memcached', 'text');
        }
        
        set_flash('success', 'Memcached-Einstellungen gespeichert.');
        redirect('/admin/performance_settings.php');
        exit;
    }
    
    // Set cache driver
    if ($action === 'set_cache_driver') {
        $driver = post('cache_driver');
        set_setting('cache_driver', $driver, 'text');
        
        set_flash('success', 'Cache-Treiber geändert: ' . AdvancedCacheManager::getDriverName());
        redirect('/admin/performance_settings.php');
        exit;
    }
    
    // Clear cache
    if ($action === 'clear_cache') {
        AdvancedCacheManager::init();
        AdvancedCacheManager::clear();
        
        set_flash('success', '🗑️ Cache geleert!');
        redirect('/admin/performance_settings.php');
        exit;
    }
}

// Initialize cache manager (if available)
$cacheManagerAvailable = class_exists('AdvancedCacheManager');

if ($cacheManagerAvailable) {
    AdvancedCacheManager::init();
    $stats = AdvancedCacheManager::getStats();
    $currentDriver = AdvancedCacheManager::getDriver();
} else {
    // Fallback if AdvancedCacheManager not loaded yet
    $stats = ['driver' => 'file', 'driver_name' => 'File Cache'];
    $currentDriver = 'file';
}

$flashMessages = get_flash();
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Performance Einstellungen</title>
    <?php
    $adminCssPath = APP_ROOT . '/assets/css/admin.css';
    $adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
    ?>
    <link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>">
    <style>
        .performance-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .driver-card {
            background: white;
            border: 2px solid #e5e7eb;
            border-radius: 12px;
            padding: 24px;
            transition: all 0.2s;
        }
        
        .driver-card.active {
            border-color: #10b981;
            background: #f0fdf4;
        }
        
        .driver-card.disabled {
            opacity: 0.6;
        }
        
        .driver-icon {
            font-size: 48px;
            margin-bottom: 10px;
        }
        
        .driver-status {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
            margin-bottom: 10px;
        }
        
        .status-active {
            background: #d1fae5;
            color: #065f46;
        }
        
        .status-available {
            background: #dbeafe;
            color: #1e40af;
        }
        
        .status-unavailable {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 20px;
        }
        
        .stat-box {
            background: #f9fafb;
            padding: 15px;
            border-radius: 8px;
            text-align: center;
        }
        
        .stat-value {
            font-size: 24px;
            font-weight: 700;
            color: #1f2937;
        }
        
        .stat-label {
            font-size: 13px;
            color: #6b7280;
            margin-top: 5px;
        }
    </style>
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <?php foreach ($flashMessages as $flash): ?>
                <div class="alert alert-<?= e($flash['type']) ?>">
                    <?= e($flash['message']) ?>
                </div>
            <?php endforeach; ?>
            
            <div class="page-header">
                <h1>⚡ Performance Einstellungen</h1>
                <p>Caching, Redis, Memcached und Object Storage</p>
            </div>
            
            <!-- Current Status -->
            <div class="content-card" style="margin-bottom: 30px; border: 2px solid #10b981;">
                <div class="card-header" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%); color: white;">
                    <h2 class="card-title" style="color: white;">📊 Aktuelle Performance</h2>
                </div>
                
                <div style="padding: 20px;">
                    <div style="display: flex; align-items: center; gap: 20px; margin-bottom: 20px;">
                        <div style="font-size: 64px;">
                            <?php
                            $icons = [
                                'file' => '📁',
                                'redis' => '🔴',
                                'memcached' => '🐘',
                                'apcu' => '⚡'
                            ];
                            echo $icons[$currentDriver] ?? '📦';
                            ?>
                        </div>
                        <div>
                            <h3 style="margin: 0 0 5px 0;">Aktiver Cache-Treiber:</h3>
                            <div style="font-size: 24px; font-weight: 700; color: #10b981;">
                                <?= e($stats['driver_name']) ?>
                            </div>
                        </div>
                    </div>
                    
                    <?php if ($currentDriver !== 'file'): ?>
                    <div class="stats-grid">
                        <?php if (isset($stats['memory_used'])): ?>
                        <div class="stat-box">
                            <div class="stat-value"><?= e($stats['memory_used']) ?></div>
                            <div class="stat-label">Memory Used</div>
                        </div>
                        <?php endif; ?>
                        
                        <?php if (isset($stats['keys'])): ?>
                        <div class="stat-box">
                            <div class="stat-value"><?= number_format($stats['keys']) ?></div>
                            <div class="stat-label">Cached Keys</div>
                        </div>
                        <?php endif; ?>
                        
                        <?php if (isset($stats['hits']) && isset($stats['misses'])): ?>
                        <?php
                        $total = $stats['hits'] + $stats['misses'];
                        $hitRate = $total > 0 ? round(($stats['hits'] / $total) * 100, 1) : 0;
                        ?>
                        <div class="stat-box">
                            <div class="stat-value"><?= $hitRate ?>%</div>
                            <div class="stat-label">Hit Rate</div>
                        </div>
                        <?php endif; ?>
                        
                        <?php if (isset($stats['uptime'])): ?>
                        <div class="stat-box">
                            <div class="stat-value"><?= gmdate("H:i", $stats['uptime']) ?></div>
                            <div class="stat-label">Uptime</div>
                        </div>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>
                    
                    <div style="margin-top: 20px;">
                        <form method="POST" style="display: inline;">
                            <?= CSRF::field() ?>
                            <input type="hidden" name="action" value="clear_cache">
                            <button type="submit" class="btn btn-danger" onclick="return confirm('Cache wirklich leeren?')">
                                🗑️ Cache leeren
                            </button>
                        </form>
                    </div>
                </div>
            </div>
            
            <!-- Cache Drivers -->
            <h2>🚀 Verfügbare Cache-Treiber</h2>
            <div class="performance-grid">
                
                <!-- Redis -->
                <div class="driver-card <?= $currentDriver === 'redis' ? 'active' : '' ?> <?= !$cacheManagerAvailable || !AdvancedCacheManager::isDriverAvailable('redis') ? 'disabled' : '' ?>">
                    <div class="driver-icon">🔴</div>
                    <h3>Redis</h3>
                    
                    <?php if ($cacheManagerAvailable && AdvancedCacheManager::isDriverAvailable('redis')): ?>
                        <span class="driver-status status-available">✓ Verfügbar</span>
                        
                        <p style="color: #6b7280; font-size: 14px; margin: 10px 0;">
                            <strong>Empfohlen für:</strong> Hohe Last, Session-Sharing, Multiple Server
                        </p>
                        
                        <p style="color: #6b7280; font-size: 13px;">
                            ✅ Ultra-schnell (In-Memory)<br>
                            ✅ Skaliert horizontal<br>
                            ✅ Session Storage<br>
                            ✅ Queue Support
                        </p>
                        
                        <button type="button" class="btn btn-sm btn-primary" onclick="document.getElementById('redis-config').style.display='block'" style="width: 100%; margin-top: 10px;">
                            ⚙️ Konfigurieren
                        </button>
                    <?php else: ?>
                        <span class="driver-status status-unavailable">✗ Nicht installiert</span>
                        
                        <p style="color: #6b7280; font-size: 14px; margin: 10px 0;">
                            <strong>Empfohlen für:</strong> Hohe Last, Session-Sharing, Multiple Server
                        </p>
                        
                        <p style="color: #6b7280; font-size: 13px;">
                            ✅ Ultra-schnell (In-Memory)<br>
                            ✅ Skaliert horizontal<br>
                            ✅ Session Storage<br>
                            ✅ Queue Support
                        </p>
                        
                        <div style="margin-top: 10px; padding: 10px; background: #fef3c7; border-radius: 6px; font-size: 13px;">
                            <strong>Installation:</strong><br>
                            <code>apt-get install php-redis</code><br>
                            <small>Dann Apache/PHP-FPM neu starten</small>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Memcached -->
                <div class="driver-card <?= $currentDriver === 'memcached' ? 'active' : '' ?> <?= !$cacheManagerAvailable || !AdvancedCacheManager::isDriverAvailable('memcached') ? 'disabled' : '' ?>">
                    <div class="driver-icon">🐘</div>
                    <h3>Memcached</h3>
                    
                    <?php if ($cacheManagerAvailable && AdvancedCacheManager::isDriverAvailable('memcached')): ?>
                        <span class="driver-status status-available">✓ Verfügbar</span>
                        
                        <p style="color: #6b7280; font-size: 14px; margin: 10px 0;">
                            <strong>Empfohlen für:</strong> Object Caching, Distributed Caching
                        </p>
                        
                        <p style="color: #6b7280; font-size: 13px;">
                            ✅ Bewährt & stabil<br>
                            ✅ Multi-Threading<br>
                            ✅ Simple Setup<br>
                            ⚠️ Keine Persistenz
                        </p>
                        
                        <button type="button" class="btn btn-sm btn-primary" onclick="document.getElementById('memcached-config').style.display='block'" style="width: 100%; margin-top: 10px;">
                            ⚙️ Konfigurieren
                        </button>
                    <?php else: ?>
                        <span class="driver-status status-unavailable">✗ Nicht installiert</span>
                        
                        <p style="color: #6b7280; font-size: 14px; margin: 10px 0;">
                            <strong>Empfohlen für:</strong> Object Caching, Distributed Caching
                        </p>
                        
                        <p style="color: #6b7280; font-size: 13px;">
                            ✅ Bewährt & stabil<br>
                            ✅ Multi-Threading<br>
                            ✅ Simple Setup<br>
                            ⚠️ Keine Persistenz
                        </p>
                        
                        <div style="margin-top: 10px; padding: 10px; background: #fef3c7; border-radius: 6px; font-size: 13px;">
                            <strong>Installation:</strong><br>
                            <code>apt-get install php-memcached</code><br>
                            <small>Dann Apache/PHP-FPM neu starten</small>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- APCu -->
                <div class="driver-card <?= $currentDriver === 'apcu' ? 'active' : '' ?> <?= !$cacheManagerAvailable || !AdvancedCacheManager::isDriverAvailable('apcu') ? 'disabled' : '' ?>">
                    <div class="driver-icon">⚡</div>
                    <h3>APCu</h3>
                    
                    <?php if ($cacheManagerAvailable && AdvancedCacheManager::isDriverAvailable('apcu')): ?>
                        <span class="driver-status status-available">✓ Verfügbar</span>
                        
                        <p style="color: #6b7280; font-size: 14px; margin: 10px 0;">
                            <strong>Empfohlen für:</strong> Single Server, Shared Hosting
                        </p>
                        
                        <p style="color: #6b7280; font-size: 13px;">
                            ✅ Sehr schnell<br>
                            ✅ Kein Setup nötig<br>
                            ✅ Low Overhead<br>
                            ⚠️ Nur Single-Server
                        </p>
                        
                        <form method="POST" style="margin-top: 10px;">
                            <?= CSRF::field() ?>
                            <input type="hidden" name="action" value="set_cache_driver">
                            <input type="hidden" name="cache_driver" value="apcu">
                            <button type="submit" class="btn btn-sm btn-success" style="width: 100%;">
                                ✅ APCu aktivieren
                            </button>
                        </form>
                    <?php else: ?>
                        <span class="driver-status status-unavailable">✗ Nicht installiert</span>
                        
                        <p style="color: #6b7280; font-size: 14px; margin: 10px 0;">
                            <strong>Empfohlen für:</strong> Single Server, Shared Hosting
                        </p>
                        
                        <p style="color: #6b7280; font-size: 13px;">
                            ✅ Sehr schnell<br>
                            ✅ Kein Setup nötig<br>
                            ✅ Low Overhead<br>
                            ⚠️ Nur Single-Server
                        </p>
                        
                        <div style="margin-top: 10px; padding: 10px; background: #fef3c7; border-radius: 6px; font-size: 13px;">
                            <strong>Installation:</strong><br>
                            <code>apt-get install php-apcu</code><br>
                            <small>Dann Apache/PHP-FPM neu starten</small>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- File Cache -->
                <div class="driver-card <?= $currentDriver === 'file' ? 'active' : '' ?>">
                    <div class="driver-icon">📁</div>
                    <h3>File Cache</h3>
                    
                    <span class="driver-status status-active">✓ Immer verfügbar</span>
                    
                    <p style="color: #6b7280; font-size: 14px; margin: 10px 0;">
                        <strong>Standard:</strong> Fallback wenn nichts anderes verfügbar
                    </p>
                    
                    <p style="color: #6b7280; font-size: 13px;">
                        ✅ Kein Setup nötig<br>
                        ✅ Funktioniert überall<br>
                        ⚠️ Langsamer als Memory<br>
                        ⚠️ I/O Overhead
                    </p>
                    
                    <?php if ($currentDriver !== 'file'): ?>
                    <form method="POST" style="margin-top: 10px;">
                        <?= CSRF::field() ?>
                        <input type="hidden" name="action" value="set_cache_driver">
                        <input type="hidden" name="cache_driver" value="file">
                        <button type="submit" class="btn btn-sm btn-secondary" style="width: 100%;">
                            Zurück zu File Cache
                        </button>
                    </form>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Redis Configuration Modal -->
            <div id="redis-config" style="display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.5); z-index: 1000; padding: 20px; overflow-y: auto;">
                <div style="max-width: 600px; margin: 50px auto; background: white; border-radius: 12px; padding: 30px;">
                    <h2>🔴 Redis Konfiguration</h2>
                    
                    <form method="POST">
                        <?= CSRF::field() ?>
                        <input type="hidden" name="action" value="save_redis">
                        
                        <div class="form-group">
                            <label>Redis Host</label>
                            <input type="text" name="redis_host" value="<?= e(get_setting('redis_host', '127.0.0.1')) ?>" class="form-control">
                            <small>Standard: 127.0.0.1 (localhost)</small>
                        </div>
                        
                        <div class="form-group">
                            <label>Redis Port</label>
                            <input type="number" name="redis_port" value="<?= e(get_setting('redis_port', 6379)) ?>" class="form-control">
                            <small>Standard: 6379</small>
                        </div>
                        
                        <div class="form-group">
                            <label>Redis Passwort (optional)</label>
                            <input type="password" name="redis_password" value="<?= e(get_setting('redis_password', '')) ?>" class="form-control">
                            <small>Leer lassen wenn kein Passwort gesetzt</small>
                        </div>
                        
                        <div class="form-group">
                            <label>Database Number</label>
                            <input type="number" name="redis_database" value="<?= e(get_setting('redis_database', 0)) ?>" class="form-control" min="0" max="15">
                            <small>Standard: 0 (0-15 möglich)</small>
                        </div>
                        
                        <div class="form-group">
                            <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                                <input type="checkbox" name="redis_enabled" value="1" <?= get_setting('redis_enabled', false) ? 'checked' : '' ?>>
                                Redis als Cache-Treiber aktivieren
                            </label>
                        </div>
                        
                        <div style="display: flex; gap: 10px; margin-top: 20px;">
                            <button type="button" onclick="testRedisConnection()" class="btn btn-secondary">
                                🔍 Verbindung testen
                            </button>
                            <button type="submit" class="btn btn-primary">
                                💾 Speichern
                            </button>
                            <button type="button" onclick="document.getElementById('redis-config').style.display='none'" class="btn btn-secondary">
                                Abbrechen
                            </button>
                        </div>
                    </form>
                </div>
            </div>
            
            <!-- Memcached Configuration Modal -->
            <div id="memcached-config" style="display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.5); z-index: 1000; padding: 20px; overflow-y: auto;">
                <div style="max-width: 600px; margin: 50px auto; background: white; border-radius: 12px; padding: 30px;">
                    <h2>🐘 Memcached Konfiguration</h2>
                    
                    <form method="POST">
                        <?= CSRF::field() ?>
                        <input type="hidden" name="action" value="save_memcached">
                        
                        <div class="form-group">
                            <label>Memcached Host</label>
                            <input type="text" name="memcached_host" value="<?= e(get_setting('memcached_host', '127.0.0.1')) ?>" class="form-control">
                            <small>Standard: 127.0.0.1 (localhost)</small>
                        </div>
                        
                        <div class="form-group">
                            <label>Memcached Port</label>
                            <input type="number" name="memcached_port" value="<?= e(get_setting('memcached_port', 11211)) ?>" class="form-control">
                            <small>Standard: 11211</small>
                        </div>
                        
                        <div class="form-group">
                            <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                                <input type="checkbox" name="memcached_enabled" value="1" <?= get_setting('memcached_enabled', false) ? 'checked' : '' ?>>
                                Memcached als Cache-Treiber aktivieren
                            </label>
                        </div>
                        
                        <div style="display: flex; gap: 10px; margin-top: 20px;">
                            <button type="button" onclick="testMemcachedConnection()" class="btn btn-secondary">
                                🔍 Verbindung testen
                            </button>
                            <button type="submit" class="btn btn-primary">
                                💾 Speichern
                            </button>
                            <button type="button" onclick="document.getElementById('memcached-config').style.display='none'" class="btn btn-secondary">
                                Abbrechen
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </main>
    </div>
    
    <script>
    function testRedisConnection() {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <?= CSRF::field() ?>
            <input type="hidden" name="action" value="test_redis">
            <input type="hidden" name="redis_host" value="${document.querySelector('[name=redis_host]').value}">
            <input type="hidden" name="redis_port" value="${document.querySelector('[name=redis_port]').value}">
            <input type="hidden" name="redis_password" value="${document.querySelector('[name=redis_password]').value}">
        `;
        document.body.appendChild(form);
        form.submit();
    }
    
    function testMemcachedConnection() {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <?= CSRF::field() ?>
            <input type="hidden" name="action" value="test_memcached">
            <input type="hidden" name="memcached_host" value="${document.querySelector('[name=memcached_host]').value}">
            <input type="hidden" name="memcached_port" value="${document.querySelector('[name=memcached_port]').value}">
        `;
        document.body.appendChild(form);
        form.submit();
    }
    </script>
</body>
</html>