<?php
/** cms-core/public/webview/admin/reset_password.php */

require_once __DIR__ . '/../../../backend/core/init.php';
require_once __DIR__ . '/../../../backend/encryption/password_utils.php';

$error = '';
$success = '';
$validToken = false;
$token = get_param('token');

// Validate token
if ($token) {
    $reset = db()->fetchOne(
        "SELECT pr.*, u.id as user_id, u.username, u.email 
         FROM password_resets pr
         JOIN users u ON pr.user_id = u.id
         WHERE pr.token = ? AND pr.expires_at > NOW() AND pr.used = 0",
        [$token]
    );
    
    if ($reset) {
        $validToken = true;
    } else {
        $error = 'Dieser Link ist ungültig oder abgelaufen. Bitte fordern Sie einen neuen Link an.';
    }
}

// Handle form submission
if (is_post() && $validToken) {
    try {
        CSRF::require();
        
        $password = post('password');
        $passwordConfirm = post('password_confirm');
        
        // Validate
        if (empty($password)) {
            $error = 'Bitte geben Sie ein neues Passwort ein.';
        } elseif (strlen($password) < 8) {
            $error = 'Das Passwort muss mindestens 8 Zeichen lang sein.';
        } elseif ($password !== $passwordConfirm) {
            $error = 'Die Passwörter stimmen nicht überein.';
        } else {
            // Hash new password
            $passwordHash = PasswordUtils::hash($password);
            
            // Update user password
            db()->execute(
                "UPDATE users SET password_hash = ? WHERE id = ?",
                [$passwordHash, $reset['user_id']]
            );
            
            // Mark token as used
            db()->execute(
                "UPDATE password_resets SET used = 1 WHERE token = ?",
                [$token]
            );
            
            Logger::info("Password reset completed for user: {$reset['username']}");
            
            $success = 'Ihr Passwort wurde erfolgreich zurückgesetzt! Sie können sich jetzt anmelden.';
        }
        
    } catch (Exception $e) {
        Logger::error("Password reset error: " . $e->getMessage());
        $error = 'Ein Fehler ist aufgetreten. Bitte versuchen Sie es erneut.';
    }
}
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Passwort zurücksetzen - KynoCanis Admin</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .container {
            background: white;
            padding: 40px;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            max-width: 450px;
            width: 100%;
        }
        
        .logo {
            text-align: center;
            margin-bottom: 30px;
        }
        
        .logo h1 {
            color: #333;
            font-size: 28px;
            margin-bottom: 5px;
        }
        
        .logo p {
            color: #666;
            font-size: 14px;
        }
        
        .info-text {
            color: #666;
            font-size: 14px;
            line-height: 1.6;
            margin-bottom: 25px;
            text-align: center;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            color: #333;
            font-weight: 500;
            font-size: 14px;
        }
        
        input[type="password"] {
            width: 100%;
            padding: 14px;
            border: 2px solid #e5e7eb;
            border-radius: 10px;
            font-size: 15px;
            transition: all 0.2s;
        }
        
        input[type="password"]:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .password-hint {
            font-size: 12px;
            color: #666;
            margin-top: 6px;
        }
        
        .btn {
            width: 100%;
            padding: 14px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s;
        }
        
        .btn:hover {
            transform: translateY(-2px);
        }
        
        .btn:active {
            transform: translateY(0);
        }
        
        .error-message {
            background: #fee;
            color: #c33;
            padding: 12px 16px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-left: 4px solid #c33;
        }
        
        .success-message {
            background: #efe;
            color: #363;
            padding: 12px 16px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-left: 4px solid #363;
        }
        
        .back-link {
            text-align: center;
            margin-top: 20px;
        }
        
        .back-link a {
            color: #667eea;
            text-decoration: none;
            font-size: 14px;
            font-weight: 600;
        }
        
        .back-link a:hover {
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="logo">
            <h1>🔑 Neues Passwort</h1>
            <p>Setzen Sie Ihr Passwort zurück</p>
        </div>
        
        <?php if ($error): ?>
            <div class="error-message">
                <?= e($error) ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="success-message">
                <?= e($success) ?>
            </div>
            <div class="back-link">
                <a href="login.php">→ Zum Login</a>
            </div>
        <?php elseif ($validToken): ?>
            <p class="info-text">
                Geben Sie Ihr neues Passwort ein.
            </p>
            
            <form method="POST">
                <?= CSRF::field() ?>
                
                <div class="form-group">
                    <label for="password">Neues Passwort</label>
                    <input 
                        type="password" 
                        id="password" 
                        name="password" 
                        required 
                        minlength="8"
                        autofocus
                    >
                    <p class="password-hint">Mindestens 8 Zeichen</p>
                </div>
                
                <div class="form-group">
                    <label for="password_confirm">Passwort bestätigen</label>
                    <input 
                        type="password" 
                        id="password_confirm" 
                        name="password_confirm" 
                        required 
                        minlength="8"
                    >
                </div>
                
                <button type="submit" class="btn">
                    🔒 Passwort zurücksetzen
                </button>
            </form>
            
            <div class="back-link">
                <a href="login.php">← Zurück zum Login</a>
            </div>
        <?php else: ?>
            <div class="back-link">
                <a href="forgot_password.php">← Neuen Link anfordern</a>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>