<?php
/** cms-core/public/webview/admin/robots_manage.php */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';

$error = '';
$success = '';
$robotsPath = APP_ROOT . '/robots.txt';
$robotsExists = file_exists($robotsPath);
$isProduction = defined('APP_ENV') && APP_ENV === 'production';

// Default robots.txt content
$defaultProduction = "User-agent: *
Allow: /

# Sitemap
Sitemap: " . APP_URL . "/sitemap.xml

# Block admin area
Disallow: /cms-core/
Disallow: /admin/

# Block private directories
Disallow: /backend/
Disallow: /config/";

$defaultDevelopment = "User-agent: *
Disallow: /

# Development environment - blocking all crawlers";

// Get current content
$currentContent = $robotsExists ? file_get_contents($robotsPath) : '';

// Handle form submission
if (is_post()) {
    try {
        CSRF::require();
        
        $action = post('action');
        
        if ($action === 'save') {
            $content = post('robots_content');
            
            if (empty($content)) {
                $error = 'Der Inhalt darf nicht leer sein.';
            } else {
                if (file_put_contents($robotsPath, $content)) {
                    Logger::info("robots.txt updated by admin");
                    set_flash('success', '✅ robots.txt erfolgreich gespeichert!');
                    redirect('/admin/robots_manage.php');
                    exit;
                } else {
                    $error = 'Fehler beim Schreiben der Datei.';
                }
            }
        }
        
        if ($action === 'reset_production') {
            if (file_put_contents($robotsPath, $defaultProduction)) {
                set_flash('success', '✅ robots.txt auf Produktions-Standard zurückgesetzt!');
                redirect('/admin/robots_manage.php');
                exit;
            }
        }
        
        if ($action === 'reset_development') {
            if (file_put_contents($robotsPath, $defaultDevelopment)) {
                set_flash('success', '✅ robots.txt auf Entwicklungs-Standard zurückgesetzt!');
                redirect('/admin/robots_manage.php');
                exit;
            }
        }
        
        if ($action === 'delete') {
            if ($robotsExists) {
                unlink($robotsPath);
                set_flash('success', 'robots.txt erfolgreich gelöscht.');
                redirect('/admin/robots_manage.php');
                exit;
            }
        }
        
    } catch (Exception $e) {
        $error = 'Fehler: ' . $e->getMessage();
    }
}

$flashMessages = get_flash();
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>robots.txt</title>
<?php
$adminCssPath = APP_ROOT . '/assets/css/admin.css';
$adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
?>
<link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>">
<style>
    .robots-textarea {
        width: 100%;
        min-height: 400px;
        font-family: 'Courier New', monospace;
        font-size: 14px;
        padding: 15px;
        border: 2px solid #e5e7eb;
        border-radius: 8px;
        resize: vertical;
    }
    .robots-textarea:focus {
        outline: none;
        border-color: #667eea;
    }
    .env-badge {
        display: inline-block;
        padding: 6px 12px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 600;
        text-transform: uppercase;
    }
    .env-production {
        background: #dcfce7;
        color: #166534;
    }
    .env-development {
        background: #fef3c7;
        color: #92400e;
    }
    .button-group {
        display: flex;
        gap: 10px;
        margin-top: 15px;
        flex-wrap: wrap;
    }
</style>
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <?php foreach ($flashMessages as $flash): ?>
                <div class="alert alert-<?= e($flash['type']) ?>">
                    <?= e($flash['message']) ?>
                </div>
            <?php endforeach; ?>
            
            <?php if ($error): ?>
                <div class="alert alert-error"><?= e($error) ?></div>
            <?php endif; ?>
            
            <div class="page-header">
                <div>
                    <h1>🤖 robots.txt Manager</h1>
                    <p class="page-description">Steuere, wie Suchmaschinen deine Website crawlen</p>
                </div>
                <div>
                    <span class="env-badge <?= $isProduction ? 'env-production' : 'env-development' ?>">
                        <?= $isProduction ? '🟢 Production' : '🟡 Development' ?>
                    </span>
                </div>
            </div>
            
            <!-- Warning Card -->
            <?php if (!$isProduction): ?>
                <div class="alert alert-warning">
                    <strong>⚠️ Entwicklungsmodus aktiv!</strong><br>
                    Im Entwicklungsmodus wird empfohlen, alle Crawler zu blockieren. 
                    Setzen Sie <code>APP_ENV=production</code> in der .env-Datei, wenn die Website live geht.
                </div>
            <?php endif; ?>
            
            <!-- Status Card -->
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">Status</h2>
                </div>
                
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px;">
                    <div style="padding: 20px; background: <?= $robotsExists ? '#efe' : '#fee' ?>; border-radius: 8px; text-align: center;">
                        <div style="font-size: 2rem; margin-bottom: 10px;">
                            <?= $robotsExists ? '✅' : '❌' ?>
                        </div>
                        <div style="font-weight: 600; color: <?= $robotsExists ? '#363' : '#c33' ?>;">
                            <?= $robotsExists ? 'robots.txt existiert' : 'Keine robots.txt' ?>
                        </div>
                    </div>
                    
                    <div style="padding: 20px; background: #f0f9ff; border-radius: 8px; text-align: center;">
                        <div style="font-size: 2rem; margin-bottom: 10px;">🌐</div>
                        <div style="font-weight: 600; color: #0369a1;">Öffentlich erreichbar</div>
                        <div style="font-size: 0.85rem; color: #64748b;">
                            <a href="<?= APP_URL ?>/robots.txt" target="_blank" style="color: #0284c7;">
                                <?= APP_URL ?>/robots.txt
                            </a>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Editor Card -->
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">robots.txt bearbeiten</h2>
                </div>
                
                <form method="POST" id="main-form">
                    <?= CSRF::field() ?>
                    <input type="hidden" name="action" value="save">
                    
                    <textarea 
                        name="robots_content" 
                        id="robots_content"
                        class="robots-textarea"
                        placeholder="User-agent: *&#10;Disallow: /"
                    ><?= e($robotsExists ? $currentContent : ($isProduction ? $defaultProduction : $defaultDevelopment)) ?></textarea>
                    
                    <div class="button-group">
                        <button type="submit" class="btn btn-primary">
                            💾 Speichern
                        </button>
                        
                        <button type="button" class="btn btn-secondary" onclick="resetProduction()">
                            🟢 Standard (Production)
                        </button>
                        
                        <button type="button" class="btn btn-secondary" onclick="resetDevelopment()">
                            🟡 Standard (Development)
                        </button>
                        
                        <?php if ($robotsExists): ?>
                            <button type="button" class="btn btn-danger" onclick="deleteRobots()">
                                🗑️ Löschen
                            </button>
                        <?php endif; ?>
                    </div>
                </form>
                
                <!-- Hidden forms for reset/delete actions -->
                <form method="POST" id="reset-production-form" style="display: none;">
                    <?= CSRF::field() ?>
                    <input type="hidden" name="action" value="reset_production">
                </form>
                
                <form method="POST" id="reset-development-form" style="display: none;">
                    <?= CSRF::field() ?>
                    <input type="hidden" name="action" value="reset_development">
                </form>
                
                <form method="POST" id="delete-form" style="display: none;">
                    <?= CSRF::field() ?>
                    <input type="hidden" name="action" value="delete">
                </form>
            </div>
            
            <!-- Info Card -->
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">ℹ️ Was ist robots.txt?</h2>
                </div>
                
                <p style="margin-bottom: 15px; line-height: 1.6;">
                    Die robots.txt-Datei teilt Suchmaschinen-Crawlern mit, welche Bereiche Ihrer Website sie durchsuchen dürfen und welche nicht.
                </p>
                
                <h3 style="margin: 20px 0 10px 0; font-size: 16px;">📋 Häufige Anwendungsfälle:</h3>
                
                <ul style="margin-left: 20px; line-height: 1.8;">
                    <li><strong>Entwicklung:</strong> Alle Crawler blockieren (<code>Disallow: /</code>)</li>
                    <li><strong>Admin-Bereich:</strong> Verwaltungsseiten blockieren (<code>Disallow: /admin/</code>)</li>
                    <li><strong>Private Ordner:</strong> Backend und Config blockieren</li>
                    <li><strong>Sitemap:</strong> Sitemap-URL angeben für bessere Indexierung</li>
                </ul>
                
                <div style="margin-top: 20px; padding: 15px; background: #fef3c7; border-left: 4px solid #f59e0b; border-radius: 4px;">
                    <strong>⚠️ Wichtig:</strong> robots.txt ist KEINE Sicherheitsmaßnahme! Dateien sind weiterhin öffentlich erreichbar, 
                    wenn jemand die URL kennt. Verwenden Sie robots.txt nur für SEO-Zwecke.
                </div>
                
                <h3 style="margin: 20px 0 10px 0; font-size: 16px;">🔍 Beispiel-Syntax:</h3>
                
                <pre style="background: #f3f4f6; padding: 15px; border-radius: 8px; overflow-x: auto; font-size: 13px;">User-agent: *
Allow: /

Disallow: /admin/
Disallow: /cms-core/

Sitemap: <?= APP_URL ?>/sitemap.xml</pre>
            </div>
        </main>
    </div>
    
    <script src="/assets/js/main.js"></script>
    <script>
    function resetProduction() {
        if (confirm('Auf Produktions-Standard zurücksetzen?')) {
            document.getElementById('reset-production-form').submit();
        }
    }
    
    function resetDevelopment() {
        if (confirm('Auf Entwicklungs-Standard zurücksetzen?')) {
            document.getElementById('reset-development-form').submit();
        }
    }
    
    function deleteRobots() {
        if (confirm('robots.txt wirklich löschen?')) {
            document.getElementById('delete-form').submit();
        }
    }
    </script>
</body>
</html>