<?php
/** cms-core/public/webview/admin/settings.php */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';
require_once __DIR__ . '/../../../backend/settings/settings_manager.php';
require_once __DIR__ . '/../../../backend/settings/site_info_manager.php';

$errors = [];

// Handle Form Submit
if (is_post()) {
    try {
        CSRF::require();
    } catch (Exception $e) {
        set_flash('error', 'Ungültiger CSRF-Token oder Serverfehler.');
        redirect('/cms-core/public/webview/admin/settings.php');
    }
    
    // Handle Logo Upload
    $logoPath = trim(post('site_logo') ?? '');
    
    if (isset($_FILES['logo_upload']) && $_FILES['logo_upload']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['logo_upload'];
        
        // Validate
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/svg+xml'];
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        if (!in_array($mimeType, $allowedTypes)) {
            $errors['logo_upload'] = 'Ungültiger Dateityp. Nur JPG, PNG, GIF, WebP und SVG erlaubt.';
        } elseif ($file['size'] > 5 * 1024 * 1024) {
            $errors['logo_upload'] = 'Datei zu groß. Maximum 5MB.';
        } else {
            // Create logo directory
            $logoDir = MEDIA_ROOT . '/logo';
            if (!is_dir($logoDir)) {
                mkdir($logoDir, 0755, true);
            }

            // Save file
            $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = 'logo_' . time() . '.' . $ext;

            if (move_uploaded_file($file['tmp_name'], $logoDir . '/' . $filename)) {
                $logoPath = 'logo/' . $filename;
            } else {
                $errors['logo_upload'] = 'Fehler beim Speichern der Datei.';
            }
        }
    }
    
    // Handle Favicon Upload
    $faviconPath = trim(post('site_favicon') ?? '');
    
    if (isset($_FILES['favicon_upload']) && $_FILES['favicon_upload']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['favicon_upload'];
        
        // Validate - Favicon should be ICO, PNG, or SVG
        $allowedTypes = ['image/x-icon', 'image/vnd.microsoft.icon', 'image/png', 'image/svg+xml'];
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        if (!in_array($mimeType, $allowedTypes)) {
            $errors['favicon_upload'] = 'Ungültiger Dateityp. Nur ICO, PNG oder SVG erlaubt.';
        } elseif ($file['size'] > 1 * 1024 * 1024) {
            $errors['favicon_upload'] = 'Datei zu groß. Maximum 1MB für Favicon.';
        } else {
            // Create favicon directory
            $faviconDir = MEDIA_ROOT . '/favicon';
            if (!is_dir($faviconDir)) {
                mkdir($faviconDir, 0755, true);
            }

            // Save file
            $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = 'favicon_' . time() . '.' . $ext;

            if (move_uploaded_file($file['tmp_name'], $faviconDir . '/' . $filename)) {
                $faviconPath = 'favicon/' . $filename;
            } else {
                $errors['favicon_upload'] = 'Fehler beim Speichern des Favicons.';
            }
        }
    }
    
    // Combine all settings
    $allSettings = [
        // Basic System
        'site_name' => trim(post('site_name') ?? ''),
        'site_email' => trim(post('site_email') ?? ''),
        'maintenance_mode' => post('maintenance_mode') === '1',
        'blog_enabled' => post('blog_enabled') === '1',
        
        // Website Info
        'site_description' => trim(post('site_description') ?? ''),
        'use_logo' => post('use_logo') === '1',
        'site_logo' => $logoPath,
        
        // Favicon Settings
        'use_logo_as_favicon' => post('use_logo_as_favicon') === '1',
        'site_favicon' => $faviconPath,
        
        // Contact Information
        'contact_email' => trim(post('contact_email') ?? ''),
        'contact_phone' => trim(post('contact_phone') ?? ''),
        'contact_address_line1' => trim(post('contact_address_line1') ?? ''),
        'contact_address_line2' => trim(post('contact_address_line2') ?? ''),
        'contact_opening_hours' => trim(post('contact_opening_hours') ?? ''),
        
        // Footer Settings
        'footer_show_address' => post('footer_show_address') === '1',
        'footer_company_description' => trim(post('footer_company_description') ?? ''),
        
        // Social Media
        'social_facebook' => trim(post('social_facebook') ?? ''),
        'social_instagram' => trim(post('social_instagram') ?? ''),
        'social_linkedin' => trim(post('social_linkedin') ?? ''),
        'social_youtube' => trim(post('social_youtube') ?? ''),
        'social_twitter' => trim(post('social_twitter') ?? ''),
    ];
    
    // Validation
    if (empty($errors)) {
        $validation1 = SettingsManager::validate([
            'site_name' => $allSettings['site_name'],
            'site_email' => $allSettings['site_email']
        ]);
        
        $validation2 = SiteInfoManager::validate($allSettings);
        
        $errors = array_merge($validation1['errors'] ?? [], $validation2['errors'] ?? []);
    }
    
    if (empty($errors)) {
        // Save basic settings
        SettingsManager::saveMultiple([
            'site_name' => $allSettings['site_name'],
            'site_email' => $allSettings['site_email'],
            'maintenance_mode' => $allSettings['maintenance_mode'],
            'blog_enabled' => $allSettings['blog_enabled']
        ]);
        
        SiteInfoManager::save($allSettings);
        
        // Handle SSL/HTTPS Setting Update in .env
        $forceSsl = post('force_ssl') === '1';
        updateEnvVariable('FORCE_SSL', $forceSsl ? 'true' : 'false');
        
        // Update APP_URL protocol if needed
        $currentUrl = defined('APP_URL') ? APP_URL : '';
        if ($currentUrl) {
            $newUrl = preg_replace('#^https?://#', '', $currentUrl);
            $protocol = $forceSsl ? 'https://' : 'http://';
            updateEnvVariable('APP_URL', $protocol . $newUrl);
        }
        
        set_flash('success', 'Alle Einstellungen erfolgreich gespeichert!');
        redirect('/cms-core/public/webview/admin/settings.php');
    }
}

// Get current settings
$currentSettings = SettingsManager::getAll();
$siteInfo = SiteInfoManager::getAll();

// Read current FORCE_SSL setting from .env
$envFile = APP_ROOT . '/.env';
$forceSsl = false;
if (file_exists($envFile)) {
    $envContent = file_get_contents($envFile);
    if (preg_match('/FORCE_SSL\s*=\s*(true|false)/i', $envContent, $matches)) {
        $forceSsl = strtolower($matches[1]) === 'true';
    }
}

$flashMessages = get_flash();

/**
 * Helper function to update .env variables
 */
function updateEnvVariable($key, $value) {
    $envFile = APP_ROOT . '/.env';
    
    if (!file_exists($envFile)) {
        return false;
    }
    
    $envContent = file_get_contents($envFile);
    
    // Check if key exists
    if (preg_match("/^{$key}\s*=.*$/m", $envContent)) {
        // Update existing
        $envContent = preg_replace("/^{$key}\s*=.*$/m", "{$key}={$value}", $envContent);
    } else {
        // Add new
        $envContent .= "\n{$key}={$value}\n";
    }
    
    return file_put_contents($envFile, $envContent) !== false;
}
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Einstellungen</title>
<?php
$adminCssPath = APP_ROOT . '/assets/css/admin.css';
$adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
?>
<link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>">
<style>
    .two-column-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 20px;
    }
    @media (max-width: 768px) {
        .two-column-grid {
            grid-template-columns: 1fr;
        }
    }
    .logo-preview, .favicon-preview {
        margin: 20px 0;
        padding: 20px;
        background: #f9fafb;
        border-radius: 8px;
        text-align: center;
        border: 2px dashed #e5e7eb;
    }
    .logo-image {
        max-width: 300px;
        max-height: 150px;
        object-fit: contain;
        border: 1px solid #e5e7eb;
        border-radius: 4px;
        padding: 10px;
        background: white;
    }
    .favicon-image {
        max-width: 64px;
        max-height: 64px;
        object-fit: contain;
        border: 1px solid #e5e7eb;
        border-radius: 4px;
        padding: 5px;
        background: white;
    }
    .hidden {
        display: none !important;
    }
    .section-divider {
        border: none;
        border-top: 1px solid #e5e7eb;
        margin: 30px 0;
    }
    .ssl-info-box {
        background: #eff6ff;
        border: 2px solid #3b82f6;
        border-radius: 8px;
        padding: 16px;
        margin: 20px 0;
    }
    .ssl-info-box h4 {
        color: #1e40af;
        margin-bottom: 8px;
        font-size: 15px;
    }
    .ssl-info-box p {
        color: #1e3a8a;
        font-size: 13px;
        line-height: 1.6;
        margin: 0;
    }
    .favicon-info-box {
        background: #fef3c7;
        border: 2px solid #f59e0b;
        border-radius: 8px;
        padding: 12px;
        margin: 15px 0;
        font-size: 13px;
        color: #92400e;
    }
    </style>
</head>
<body>
    <?php include __DIR__ . '/includes/admin_header.php'; ?>
    
    <div class="admin-container">
        <?php include __DIR__ . '/includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <?php foreach ($flashMessages as $flash): ?>
                <div class="alert alert-<?= e($flash['type']) ?>">
                    <?= e($flash['message']) ?>
                </div>
            <?php endforeach; ?>
            
            <div class="page-header">
                <div>
                    <h1>⚙️ Website-Einstellungen</h1>
                    <p class="page-description">Grundlegende Konfiguration Ihrer Website</p>
                </div>
            </div>
            
            <form method="POST" enctype="multipart/form-data">
                <?= CSRF::field() ?>
                
                <!-- Basic Settings -->
                <div class="content-card">
                    <div class="card-header">
                        <h2 class="card-title">🌐 Grundeinstellungen</h2>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="site_name">Website-Name *</label>
                        <input 
                            type="text" 
                            id="site_name" 
                            name="site_name" 
                            class="form-input"
                            value="<?= e($currentSettings['site_name'] ?? '') ?>"
                            required
                        >
                        <?php if (isset($errors['site_name'])): ?>
                            <span class="form-error"><?= e($errors['site_name']) ?></span>
                        <?php endif; ?>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="site_email">System-E-Mail *</label>
                        <input 
                            type="email" 
                            id="site_email" 
                            name="site_email" 
                            class="form-input"
                            value="<?= e($currentSettings['site_email'] ?? '') ?>"
                            required
                        >
                        <?php if (isset($errors['site_email'])): ?>
                            <span class="form-error"><?= e($errors['site_email']) ?></span>
                        <?php endif; ?>
                        <small class="form-hint">Wird für System-E-Mails verwendet</small>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="site_description">Website-Beschreibung (SEO)</label>
                        <textarea 
                            id="site_description" 
                            name="site_description" 
                            class="form-textarea"
                            rows="2"
                            maxlength="160"
                            placeholder="Kurze Beschreibung für Suchmaschinen (max. 160 Zeichen)"
                        ><?= e($siteInfo['site_description'] ?? '') ?></textarea>
                        <small class="form-hint">Standard Meta-Description für Suchmaschinen</small>
                    </div>
                    
                    <hr class="section-divider">
                    
                    <!-- SSL/HTTPS Settings -->
                    <div class="ssl-info-box">
                        <h4>🔒 SSL/HTTPS Einstellungen</h4>
                        <p>Steuert, ob alle Links auf Ihrer Website mit HTTPS (sicher) oder HTTP (unsicher) generiert werden. Aktivieren Sie dies nur, wenn Sie ein gültiges SSL-Zertifikat haben.</p>
                    </div>
                    
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input 
                                type="checkbox" 
                                name="force_ssl" 
                                value="1"
                                <?= $forceSsl ? 'checked' : '' ?>
                            >
                            <span>
                                <strong>🔐 SSL/HTTPS erzwingen</strong>
                                <small class="form-hint" style="display: block;">Alle Links werden mit HTTPS generiert. Deaktivieren Sie dies, wenn Ihre Website kein SSL-Zertifikat hat (dann werden HTTP-Links verwendet).</small>
                            </span>
                        </label>
                    </div>
                    
                    <hr class="section-divider">
                    
                    <!-- Logo Checkbox -->
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input 
                                type="checkbox" 
                                id="use_logo_checkbox"
                                name="use_logo" 
                                value="1"
                                <?= !empty($siteInfo['use_logo']) ? 'checked' : '' ?>
                                onchange="toggleLogoUpload()"
                            >
                            <span>
                                <strong>🖼️ Logo verwenden</strong>
                                <small class="form-hint" style="display: block;">Logo statt Website-Name im Header anzeigen</small>
                            </span>
                        </label>
                    </div>
                    
                    <!-- Logo Upload Section -->
                    <div id="logo-upload-section" class="<?= empty($siteInfo['use_logo']) ? 'hidden' : '' ?>">
                        <div class="form-group">
                            <label class="form-label">Logo hochladen</label>
                            
                            <!-- Current Logo Preview -->
                            <?php if (!empty($siteInfo['site_logo'])): ?>
                                <div class="logo-preview" id="logo-preview-container">
                                    <p class="form-hint">Aktuelles Logo:</p>
                                    <img src="/cms-core/public/webview/media.php?file=<?= e($siteInfo['site_logo']) ?>" 
                                         alt="Logo" 
                                         class="logo-image"
                                         id="current-logo-preview">
                                    <button type="button" 
                                            class="btn btn-sm btn-danger" 
                                            onclick="removeLogo()"
                                            style="margin-top: 10px;">
                                        🗑️ Logo entfernen
                                    </button>
                                </div>
                            <?php endif; ?>
                            
                            <!-- Error Message -->
                            <?php if (isset($errors['logo_upload'])): ?>
                                <div class="alert alert-error">
                                    <?= e($errors['logo_upload']) ?>
                                </div>
                            <?php endif; ?>
                            
                            <!-- Hidden field to store logo path -->
                            <input type="hidden" 
                                   id="site_logo" 
                                   name="site_logo" 
                                   value="<?= e($siteInfo['site_logo'] ?? '') ?>">
                            
                            <!-- Upload Buttons -->
                            <div class="logo-upload-actions" style="display: flex; gap: 10px; margin-top: 15px; flex-wrap: wrap;">
                                <label class="btn btn-primary" style="cursor: pointer; margin: 0;">
                                    📤 Neue Datei hochladen
                                    <input type="file" 
                                           name="logo_upload" 
                                           accept="image/*"
                                           style="display: none;"
                                           onchange="previewLogo(this)">
                                </label>
                                
                                <button type="button" 
                                        class="btn btn-secondary" 
                                        onclick="openMediaSelector('logo')">
                                    📁 Aus Mediathek wählen
                                </button>
                            </div>
                            
                            <small class="form-hint" style="display: block; margin-top: 10px;">
                                Empfohlene Größe: max. 200px Höhe, transparenter Hintergrund (PNG) • Max. 5MB
                            </small>
                        </div>
                    </div>
                    
                    <hr class="section-divider">
                    
                    <!-- Favicon Section -->
                    <div class="form-group">
                        <label class="form-label">🌟 Favicon (Browser-Icon)</label>
                        
                        <div class="favicon-info-box">
                            💡 <strong>Tipp:</strong> Ein Favicon ist das kleine Icon, das in Browser-Tabs angezeigt wird. Empfohlen: 32x32px oder 64x64px, PNG/ICO Format.
                        </div>
                        
                        <!-- Use Logo as Favicon Option -->
                        <label class="checkbox-label" style="margin-bottom: 15px;">
                            <input 
                                type="checkbox" 
                                id="use_logo_as_favicon_checkbox"
                                name="use_logo_as_favicon" 
                                value="1"
                                <?= !empty($siteInfo['use_logo_as_favicon']) ? 'checked' : '' ?>
                                onchange="toggleFaviconUpload()"
                            >
                            <span>
                                <strong>Logo als Favicon verwenden</strong>
                                <small class="form-hint" style="display: block;">Das hochgeladene Logo wird automatisch als Favicon verwendet</small>
                            </span>
                        </label>
                        
                        <!-- Custom Favicon Upload Section -->
                        <div id="favicon-upload-section" class="<?= !empty($siteInfo['use_logo_as_favicon']) ? 'hidden' : '' ?>">
                            <!-- Current Favicon Preview -->
                            <?php if (!empty($siteInfo['site_favicon'])): ?>
                                <div class="favicon-preview" id="favicon-preview-container">
                                    <p class="form-hint">Aktuelles Favicon:</p>
                                    <img src="/cms-core/public/webview/media.php?file=<?= e($siteInfo['site_favicon']) ?>" 
                                         alt="Favicon" 
                                         class="favicon-image"
                                         id="current-favicon-preview">
                                    <button type="button" 
                                            class="btn btn-sm btn-danger" 
                                            onclick="removeFavicon()"
                                            style="margin-top: 10px;">
                                        🗑️ Favicon entfernen
                                    </button>
                                </div>
                            <?php endif; ?>
                            
                            <!-- Error Message -->
                            <?php if (isset($errors['favicon_upload'])): ?>
                                <div class="alert alert-error">
                                    <?= e($errors['favicon_upload']) ?>
                                </div>
                            <?php endif; ?>
                            
                            <!-- Hidden field to store favicon path -->
                            <input type="hidden" 
                                   id="site_favicon" 
                                   name="site_favicon" 
                                   value="<?= e($siteInfo['site_favicon'] ?? '') ?>">
                            
                            <!-- Upload Buttons -->
                            <div class="favicon-upload-actions" style="display: flex; gap: 10px; margin-top: 15px; flex-wrap: wrap;">
                                <label class="btn btn-primary" style="cursor: pointer; margin: 0;">
                                    📤 Favicon hochladen
                                    <input type="file" 
                                           name="favicon_upload" 
                                           accept=".ico,.png,.svg,image/x-icon,image/png,image/svg+xml"
                                           style="display: none;"
                                           onchange="previewFavicon(this)">
                                </label>
                                
                                <button type="button" 
                                        class="btn btn-secondary" 
                                        onclick="openMediaSelector('favicon')">
                                    📁 Aus Mediathek wählen
                                </button>
                            </div>
                            
                            <small class="form-hint" style="display: block; margin-top: 10px;">
                                Empfohlene Größe: 32x32px oder 64x64px • Formate: ICO, PNG, SVG • Max. 1MB
                            </small>
                        </div>
                    </div>
                    
                    <hr class="section-divider">
                    
                    <!-- Blog Enable/Disable -->
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input 
                                type="checkbox" 
                                name="blog_enabled" 
                                value="1"
                                <?= !empty($currentSettings['blog_enabled']) ? 'checked' : '' ?>
                            >
                            <span>
                                <strong>📝 Blog-Funktion aktivieren</strong>
                                <small class="form-hint" style="display: block;">Blog-Bereich in Navigation und Footer anzeigen</small>
                            </span>
                        </label>
                    </div>
                    
                    <hr class="section-divider">
                    
                    <!-- Maintenance Mode -->
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input 
                                type="checkbox" 
                                name="maintenance_mode" 
                                value="1"
                                <?= !empty($currentSettings['maintenance_mode']) ? 'checked' : '' ?>
                            >
                            <span>
                                <strong>🔧 Wartungsmodus aktivieren</strong>
                                <small class="form-hint" style="display: block;">Website für Besucher deaktivieren (Admins können weiterhin zugreifen)</small>
                            </span>
                        </label>
                    </div>
                </div>
                
                <!-- Contact Information -->
                <div class="content-card">
                    <div class="card-header">
                        <h2 class="card-title">📞 Kontaktinformationen</h2>
                        <p class="form-hint">Werden auf der Kontaktseite und im Footer angezeigt</p>
                    </div>
                    
                    <div class="two-column-grid">
                        <div class="form-group">
                            <label class="form-label" for="contact_email">Kontakt-E-Mail *</label>
                            <input 
                                type="email" 
                                id="contact_email" 
                                name="contact_email" 
                                class="form-input"
                                value="<?= e($siteInfo['contact_email'] ?? '') ?>"
                                required
                            >
                            <?php if (isset($errors['contact_email'])): ?>
                                <span class="form-error"><?= e($errors['contact_email']) ?></span>
                            <?php endif; ?>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="contact_phone">Telefon</label>
                            <input 
                                type="text" 
                                id="contact_phone" 
                                name="contact_phone" 
                                class="form-input"
                                value="<?= e($siteInfo['contact_phone'] ?? '') ?>"
                                placeholder="z.B. +49 123 456789"
                            >
                            <?php if (isset($errors['contact_phone'])): ?>
                                <span class="form-error"><?= e($errors['contact_phone']) ?></span>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="two-column-grid">
                        <div class="form-group">
                            <label class="form-label" for="contact_address_line1">Adresse Zeile 1</label>
                            <input 
                                type="text" 
                                id="contact_address_line1" 
                                name="contact_address_line1" 
                                class="form-input"
                                value="<?= e($siteInfo['contact_address_line1'] ?? '') ?>"
                                placeholder="z.B. Musterstraße 123"
                            >
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="contact_address_line2">Adresse Zeile 2</label>
                            <input 
                                type="text" 
                                id="contact_address_line2" 
                                name="contact_address_line2" 
                                class="form-input"
                                value="<?= e($siteInfo['contact_address_line2'] ?? '') ?>"
                                placeholder="z.B. 12345 Musterstadt"
                            >
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="contact_opening_hours">Öffnungszeiten</label>
                        <textarea 
                            id="contact_opening_hours" 
                            name="contact_opening_hours" 
                            class="form-textarea"
                            rows="3"
                            placeholder="z.B.&#10;Mo-Fr: 9:00 - 18:00 Uhr&#10;Sa: 10:00 - 14:00 Uhr"
                        ><?= e($siteInfo['contact_opening_hours'] ?? '') ?></textarea>
                    </div>
                </div>
                
                <!-- Footer Settings -->
                <div class="content-card">
                    <div class="card-header">
                        <h2 class="card-title">🦶 Footer-Einstellungen</h2>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="footer_company_description">Firmenbeschreibung</label>
                        <textarea 
                            id="footer_company_description" 
                            name="footer_company_description" 
                            class="form-textarea"
                            rows="3"
                            placeholder="Kurze Beschreibung Ihres Unternehmens für den Footer"
                        ><?= e($siteInfo['footer_company_description'] ?? '') ?></textarea>
                        <small class="form-hint">Wird im Footer neben dem Logo/Namen angezeigt</small>
                    </div>
                    
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input 
                                type="checkbox" 
                                name="footer_show_address" 
                                value="1"
                                <?= !empty($siteInfo['footer_show_address']) ? 'checked' : '' ?>
                            >
                            <span>
                                <strong>Adresse im Footer anzeigen</strong>
                            </span>
                        </label>
                    </div>
                </div>
                
                <!-- Social Media -->
                <div class="content-card">
                    <div class="card-header">
                        <h2 class="card-title">📱 Social Media</h2>
                        <p class="form-hint">Vollständige URLs zu Ihren Social-Media-Profilen</p>
                    </div>
                    
                    <div class="two-column-grid">
                        <div class="form-group">
                            <label class="form-label" for="social_facebook">📘 Facebook</label>
                            <input 
                                type="url" 
                                id="social_facebook" 
                                name="social_facebook" 
                                class="form-input"
                                value="<?= e($siteInfo['social_facebook'] ?? '') ?>"
                                placeholder="https://facebook.com/..."
                            >
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="social_instagram">📷 Instagram</label>
                            <input 
                                type="url" 
                                id="social_instagram" 
                                name="social_instagram" 
                                class="form-input"
                                value="<?= e($siteInfo['social_instagram'] ?? '') ?>"
                                placeholder="https://instagram.com/..."
                            >
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="social_linkedin">💼 LinkedIn</label>
                            <input 
                                type="url" 
                                id="social_linkedin" 
                                name="social_linkedin" 
                                class="form-input"
                                value="<?= e($siteInfo['social_linkedin'] ?? '') ?>"
                                placeholder="https://linkedin.com/company/..."
                            >
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="social_youtube">🎥 YouTube</label>
                            <input 
                                type="url" 
                                id="social_youtube" 
                                name="social_youtube" 
                                class="form-input"
                                value="<?= e($siteInfo['social_youtube'] ?? '') ?>"
                                placeholder="https://youtube.com/..."
                            >
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="social_twitter">🐦 Twitter / X</label>
                            <input 
                                type="url" 
                                id="social_twitter" 
                                name="social_twitter" 
                                class="form-input"
                                value="<?= e($siteInfo['social_twitter'] ?? '') ?>"
                                placeholder="https://twitter.com/..."
                            >
                        </div>
                    </div>
                </div>
                
                <!-- Submit -->
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">💾 Einstellungen speichern</button>
                </div>
            </form>
        </main>
    </div>
    
    <!-- Media Selector Modal -->
    <div id="media-selector-modal" class="modal" style="display: none;">
        <div class="modal-content" style="max-width: 900px;">
            <div class="modal-header">
                <h3 id="media-modal-title">📁 Aus Mediathek wählen</h3>
                <button type="button" class="modal-close" onclick="closeMediaSelector()">×</button>
            </div>
            <div class="modal-body">
                <div id="media-library-grid" class="media-library-grid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(150px, 1fr)); gap: 15px; max-height: 500px; overflow-y: auto;">
                    <div style="grid-column: 1/-1; text-align: center; padding: 40px; color: #9ca3af;">
                        Lade Medien...
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
    let currentMediaType = 'logo'; // Track which media type we're selecting
    
    // Toggle logo upload section
    function toggleLogoUpload() {
        const checkbox = document.getElementById('use_logo_checkbox');
        const section = document.getElementById('logo-upload-section');
        
        if (checkbox && section) {
            if (checkbox.checked) {
                section.classList.remove('hidden');
            } else {
                section.classList.add('hidden');
            }
        }
    }
    
    // Toggle favicon upload section
    function toggleFaviconUpload() {
        const checkbox = document.getElementById('use_logo_as_favicon_checkbox');
        const section = document.getElementById('favicon-upload-section');
        
        if (checkbox && section) {
            if (checkbox.checked) {
                section.classList.add('hidden');
            } else {
                section.classList.remove('hidden');
            }
        }
    }
    
    // Preview logo on file select
    function previewLogo(input) {
        if (input.files && input.files[0]) {
            const reader = new FileReader();
            reader.onload = function(e) {
                let container = document.getElementById('logo-preview-container');
                let preview = document.getElementById('current-logo-preview');
                
                if (!container) {
                    const section = document.getElementById('logo-upload-section');
                    container = document.createElement('div');
                    container.id = 'logo-preview-container';
                    container.className = 'logo-preview';
                    container.innerHTML = `
                        <p class="form-hint">Vorschau:</p>
                        <img id="current-logo-preview" class="logo-image" alt="Logo Preview">
                        <button type="button" class="btn btn-sm btn-danger" onclick="removeLogo()" style="margin-top: 10px;">
                            🗑️ Logo entfernen
                        </button>
                    `;
                    section.insertBefore(container, section.querySelector('.logo-upload-actions'));
                    preview = document.getElementById('current-logo-preview');
                }
                
                if (preview) {
                    preview.src = e.target.result;
                    container.style.display = 'block';
                }
            };
            reader.readAsDataURL(input.files[0]);
        }
    }
    
    // Preview favicon on file select
    function previewFavicon(input) {
        if (input.files && input.files[0]) {
            const reader = new FileReader();
            reader.onload = function(e) {
                let container = document.getElementById('favicon-preview-container');
                let preview = document.getElementById('current-favicon-preview');
                
                if (!container) {
                    const section = document.getElementById('favicon-upload-section');
                    container = document.createElement('div');
                    container.id = 'favicon-preview-container';
                    container.className = 'favicon-preview';
                    container.innerHTML = `
                        <p class="form-hint">Vorschau:</p>
                        <img id="current-favicon-preview" class="favicon-image" alt="Favicon Preview">
                        <button type="button" class="btn btn-sm btn-danger" onclick="removeFavicon()" style="margin-top: 10px;">
                            🗑️ Favicon entfernen
                        </button>
                    `;
                    section.insertBefore(container, section.querySelector('.favicon-upload-actions'));
                    preview = document.getElementById('current-favicon-preview');
                }
                
                if (preview) {
                    preview.src = e.target.result;
                    container.style.display = 'block';
                }
            };
            reader.readAsDataURL(input.files[0]);
        }
    }
    
    // Remove logo
    function removeLogo() {
        if (confirm('Logo wirklich entfernen?')) {
            document.getElementById('site_logo').value = '';
            const container = document.getElementById('logo-preview-container');
            if (container) {
                container.remove();
            }
        }
    }
    
    // Remove favicon
    function removeFavicon() {
        if (confirm('Favicon wirklich entfernen?')) {
            document.getElementById('site_favicon').value = '';
            const container = document.getElementById('favicon-preview-container');
            if (container) {
                container.remove();
            }
        }
    }
    
    // Open media selector
    function openMediaSelector(type) {
        currentMediaType = type;
        const modal = document.getElementById('media-selector-modal');
        const title = document.getElementById('media-modal-title');
        
        if (type === 'logo') {
            title.textContent = '📁 Logo aus Mediathek wählen';
        } else {
            title.textContent = '📁 Favicon aus Mediathek wählen';
        }
        
        modal.style.display = 'flex';
        loadMediaLibrary(type);
    }
    
    // Close media selector
    function closeMediaSelector() {
        document.getElementById('media-selector-modal').style.display = 'none';
    }
    
    // Load media from library
    function loadMediaLibrary(type) {
        const grid = document.getElementById('media-library-grid');
        grid.innerHTML = '<div style="grid-column: 1/-1; text-align: center; padding: 40px; color: #9ca3af;">Lade Medien...</div>';
        
        const endpoint = type === 'logo' ? 'logo_selector.php' : 'favicon_selector.php';
        
        fetch(endpoint)
            .then(response => response.json())
            .then(data => {
                if (data.success && data.images && data.images.length > 0) {
                    grid.innerHTML = data.images.map(img => `
                        <div class="media-item" onclick="selectMedia('${img.path}', '${type}')" style="cursor: pointer; border: 2px solid #e5e7eb; border-radius: 8px; padding: 10px; transition: all 0.2s; text-align: center;" onmouseover="this.style.borderColor='#667eea'; this.style.backgroundColor='#f0f4ff'" onmouseout="this.style.borderColor='#e5e7eb'; this.style.backgroundColor='white'">
                            <img src="/cms-core/public/webview/media.php?file=${img.path}" alt="${img.filename}" style="max-width: 100%; max-height: 100px; object-fit: contain; margin-bottom: 8px;">
                            <div style="font-size: 11px; color: #6b7280; word-break: break-word;">${img.filename}</div>
                        </div>
                    `).join('');
                } else {
                    grid.innerHTML = `
                        <div style="grid-column: 1/-1; text-align: center; padding: 40px;">
                            <div style="font-size: 48px; opacity: 0.3; margin-bottom: 10px;">📁</div>
                            <div style="color: #9ca3af;">Keine ${type === 'logo' ? 'Logos' : 'Favicons'} in der Mediathek gefunden</div>
                            <div style="color: #9ca3af; font-size: 13px; margin-top: 5px;">Laden Sie zuerst eine Datei hoch</div>
                        </div>
                    `;
                }
            })
            .catch(error => {
                console.error('Error loading media:', error);
                grid.innerHTML = `
                    <div style="grid-column: 1/-1; text-align: center; padding: 40px; color: #ef4444;">
                        Fehler beim Laden der Medien
                    </div>
                `;
            });
    }
    
    // Select media from library
    function selectMedia(path, type) {
        if (type === 'logo') {
            document.getElementById('site_logo').value = path;
            
            let container = document.getElementById('logo-preview-container');
            let preview = document.getElementById('current-logo-preview');
            
            if (!container) {
                const section = document.getElementById('logo-upload-section');
                container = document.createElement('div');
                container.id = 'logo-preview-container';
                container.className = 'logo-preview';
                container.innerHTML = `
                    <p class="form-hint">Aktuelles Logo:</p>
                    <img id="current-logo-preview" class="logo-image" alt="Logo">
                    <button type="button" class="btn btn-sm btn-danger" onclick="removeLogo()" style="margin-top: 10px;">
                        🗑️ Logo entfernen
                    </button>
                `;
                section.insertBefore(container, section.querySelector('.logo-upload-actions'));
                preview = document.getElementById('current-logo-preview');
            }
            
            if (preview) {
                preview.src = '/cms-core/public/webview/media.php?file=' + path;
                container.style.display = 'block';
            }
        } else {
            document.getElementById('site_favicon').value = path;
            
            let container = document.getElementById('favicon-preview-container');
            let preview = document.getElementById('current-favicon-preview');
            
            if (!container) {
                const section = document.getElementById('favicon-upload-section');
                container = document.createElement('div');
                container.id = 'favicon-preview-container';
                container.className = 'favicon-preview';
                container.innerHTML = `
                    <p class="form-hint">Aktuelles Favicon:</p>
                    <img id="current-favicon-preview" class="favicon-image" alt="Favicon">
                    <button type="button" class="btn btn-sm btn-danger" onclick="removeFavicon()" style="margin-top: 10px;">
                        🗑️ Favicon entfernen
                    </button>
                `;
                section.insertBefore(container, section.querySelector('.favicon-upload-actions'));
                preview = document.getElementById('current-favicon-preview');
            }
            
            if (preview) {
                preview.src = '/cms-core/public/webview/media.php?file=' + path;
                container.style.display = 'block';
            }
        }
        
        closeMediaSelector();
    }
    
    // Initialize on page load
    document.addEventListener('DOMContentLoaded', function() {
        toggleLogoUpload();
        toggleFaviconUpload();
    });
    
    // Close modal on outside click
    document.getElementById('media-selector-modal')?.addEventListener('click', function(e) {
        if (e.target === this) {
            closeMediaSelector();
        }
    });
    </script>
    <!-- CRITICAL: Add main.js for mobile menu functionality -->
    <script src="/assets/js/main.js"></script>
</body>
</html>