<?php
/** cms-core/public/webview/admin/user_manage.php - FIXED VERSION */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';
require_once __DIR__ . '/../../../backend/auth/UserRoles.php';

// Check permission
UserRoles::require('users.manage', 'Sie haben keine Berechtigung, Benutzer zu verwalten.');

// Handle actions
if (is_post()) {
    CSRF::require();
    
    $action = post('action');
    
    switch ($action) {
        case 'create':
            $result = UserRoles::createUser(
                post('username'),
                post('email'),
                post('password'),
                post('role', 'editor')
            );
            
            if ($result['success']) {
                UserRoles::logActivity($_SESSION['user_id'], 'user_created', "User ID: {$result['user_id']}");
                set_flash('success', 'Benutzer erfolgreich erstellt!');
            } else {
                set_flash('error', $result['error'] ?? 'Fehler beim Erstellen.');
            }
            redirect('/admin/user_manage');
            break;
            
        case 'update':
            $userId = (int)post('user_id');
            $data = [];
            
            if (post('email')) $data['email'] = post('email');
            if (post('role')) $data['role'] = post('role');
            if (post('password')) $data['password'] = post('password');
            
            $result = UserRoles::updateUser($userId, $data);
            
            if ($result['success']) {
                UserRoles::logActivity($_SESSION['user_id'], 'user_updated', "User ID: $userId");
                set_flash('success', 'Benutzer aktualisiert!');
            } else {
                set_flash('error', $result['error'] ?? 'Fehler beim Aktualisieren.');
            }
            redirect('/admin/user_manage');
            break;
            
        case 'delete':
            $userId = (int)post('user_id');
            $result = UserRoles::deleteUser($userId);
            
            if ($result['success']) {
                UserRoles::logActivity($_SESSION['user_id'], 'user_deleted', "User ID: $userId");
                set_flash('success', 'Benutzer gelöscht!');
            } else {
                set_flash('error', $result['error'] ?? 'Fehler beim Löschen.');
            }
            redirect('/admin/user_manage');
            break;
    }
}

$users = UserRoles::getAllUsers();
$roles = UserRoles::getAllRoles();
$currentUser = UserRoles::getUser($_SESSION['user_id']);

$pageBrowserTitle = 'Benutzerverwaltung';
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageBrowserTitle) ?></title>
    <?php
    $adminCssPath = APP_ROOT . '/assets/css/admin.css';
    $adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
    ?>
    <link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>">
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <div class="page-header">
                <div>
                    <h1>👥 Benutzerverwaltung</h1>
                    <p class="page-description">Verwalten Sie Benutzer und deren Rollen</p>
                </div>
                <button onclick="showCreateUserModal()" class="btn btn-primary">
                    ➕ Neuer Benutzer
                </button>
            </div>
            
            <?php if (has_flash()): ?>
                <?php foreach (get_flash() as $flash): ?>
                <div class="alert alert-<?= $flash['type'] ?>">
                    <?= e($flash['message']) ?>
                </div>
                <?php endforeach; ?>
            <?php endif; ?>
            
            <!-- Users Table -->
            <div class="card">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Benutzer</th>
                            <th>E-Mail</th>
                            <th>Rolle</th>
                            <th>Erstellt</th>
                            <th>Letzter Login</th>
                            <th>Aktionen</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): 
                            $roleInfo = UserRoles::getRoleInfo($user['role']);
                            $isCurrentUser = $user['id'] == $_SESSION['user_id'];
                        ?>
                        <tr>
                            <td>
                                <div style="display: flex; align-items: center; gap: 10px;">
                                    <span style="font-size: 24px;"><?= $roleInfo['icon'] ?></span>
                                    <div>
                                        <strong><?= e($user['username']) ?></strong>
                                        <?php if ($isCurrentUser): ?>
                                            <span class="badge badge-primary" style="margin-left: 8px;">Sie</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </td>
                            <td><?= e($user['email']) ?></td>
                            <td>
                                <span class="badge badge-<?= $user['role'] === 'super_admin' ? 'warning' : 'info' ?>">
                                    <?= $roleInfo['name'] ?>
                                </span>
                            </td>
                            <td><?= date('d.m.Y H:i', strtotime($user['created_at'])) ?></td>
                            <td>
                                <?php if ($user['last_login']): ?>
                                    <?= date('d.m.Y H:i', strtotime($user['last_login'])) ?>
                                <?php else: ?>
                                    <span style="color: #9ca3af;">Noch nie</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <div class="action-buttons">
                                    <button onclick="editUser(<?= $user['id'] ?>)" class="btn-icon" title="Bearbeiten">
                                        ✏️
                                    </button>
                                    <?php if (!$isCurrentUser): ?>
                                    <button onclick="deleteUser(<?= $user['id'] ?>, '<?= e($user['username']) ?>')" 
                                            class="btn-icon btn-danger" title="Löschen">
                                        🗑️
                                    </button>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Role Information -->
            <div class="card" style="margin-top: 30px;">
                <h2>🔑 Verfügbare Rollen</h2>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin-top: 20px;">
                    <?php foreach ($roles as $roleKey => $roleData): ?>
                    <div style="border: 2px solid #e5e7eb; border-radius: 8px; padding: 20px;">
                        <div style="font-size: 36px; text-align: center; margin-bottom: 10px;">
                            <?= $roleData['icon'] ?>
                        </div>
                        <h3 style="text-align: center; margin-bottom: 5px;"><?= $roleData['name'] ?></h3>
                        <p style="text-align: center; color: #6b7280; font-size: 14px; margin-bottom: 15px;">
                            <?= $roleData['description'] ?>
                        </p>
                        <div style="font-size: 12px; color: #9ca3af;">
                            <strong>Berechtigungen:</strong>
                            <ul style="margin: 5px 0 0 20px; line-height: 1.6;">
                                <?php foreach (array_slice($roleData['permissions'], 0, 5) as $perm): ?>
                                <li><?= $perm ?></li>
                                <?php endforeach; ?>
                                <?php if (count($roleData['permissions']) > 5): ?>
                                <li><em>+ <?= count($roleData['permissions']) - 5 ?> weitere...</em></li>
                                <?php endif; ?>
                            </ul>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </main>
    </div>

<!-- Create User Modal -->
<div id="createUserModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h2>➕ Neuer Benutzer</h2>
            <button onclick="closeModal('createUserModal')" class="modal-close">&times;</button>
        </div>
        <form method="POST">
            <?= CSRF::field() ?>
            <input type="hidden" name="action" value="create">
            
            <div class="form-group">
                <label>Benutzername *</label>
                <input type="text" name="username" required pattern="[a-zA-Z0-9_.-]{3,20}" 
                       placeholder="3-20 Zeichen" class="form-control">
            </div>
            
            <div class="form-group">
                <label>E-Mail *</label>
                <input type="email" name="email" required class="form-control">
            </div>
            
            <div class="form-group">
                <label>Passwort *</label>
                <input type="password" name="password" required minlength="8" class="form-control">
                <small>Mindestens 8 Zeichen, 1 Groß-, 1 Kleinbuchstabe, 1 Zahl</small>
            </div>
            
            <div class="form-group">
                <label>Rolle *</label>
                <select name="role" required class="form-control">
                    <?php foreach ($roles as $roleKey => $roleData): ?>
                    <option value="<?= $roleKey ?>" <?= $roleKey === 'editor' ? 'selected' : '' ?>>
                        <?= $roleData['icon'] ?> <?= $roleData['name'] ?> - <?= $roleData['description'] ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="modal-footer">
                <button type="button" onclick="closeModal('createUserModal')" class="btn btn-secondary">Abbrechen</button>
                <button type="submit" class="btn btn-primary">Erstellen</button>
            </div>
        </form>
    </div>
</div>

<!-- Edit User Modal -->
<div id="editUserModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h2>✏️ Benutzer bearbeiten</h2>
            <button onclick="closeModal('editUserModal')" class="modal-close">&times;</button>
        </div>
        <form method="POST" id="editUserForm">
            <?= CSRF::field() ?>
            <input type="hidden" name="action" value="update">
            <input type="hidden" name="user_id" id="edit_user_id">
            
            <div class="form-group">
                <label>E-Mail</label>
                <input type="email" name="email" id="edit_email" class="form-control">
            </div>
            
            <div class="form-group">
                <label>Rolle</label>
                <select name="role" id="edit_role" class="form-control">
                    <?php foreach ($roles as $roleKey => $roleData): ?>
                    <option value="<?= $roleKey ?>">
                        <?= $roleData['icon'] ?> <?= $roleData['name'] ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="form-group">
                <label>Neues Passwort <small>(leer lassen um nicht zu ändern)</small></label>
                <input type="password" name="password" minlength="8" class="form-control">
            </div>
            
            <div class="modal-footer">
                <button type="button" onclick="closeModal('editUserModal')" class="btn btn-secondary">Abbrechen</button>
                <button type="submit" class="btn btn-primary">Speichern</button>
            </div>
        </form>
    </div>
</div>

<!-- Delete User Form -->
<form method="POST" id="deleteUserForm" style="display: none;">
    <?= CSRF::field() ?>
    <input type="hidden" name="action" value="delete">
    <input type="hidden" name="user_id" id="delete_user_id">
</form>

<style>
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 9999;
}
.modal-content {
    background: white;
    border-radius: 12px;
    width: 90%;
    max-width: 500px;
    max-height: 90vh;
    overflow-y: auto;
}
.modal-header {
    padding: 20px;
    border-bottom: 1px solid #e5e7eb;
    display: flex;
    justify-content: space-between;
    align-items: center;
}
.modal-header h2 {
    margin: 0;
}
.modal-close {
    background: none;
    border: none;
    font-size: 32px;
    cursor: pointer;
    color: #6b7280;
}
.modal-close:hover {
    color: #1f2937;
}
.modal form {
    padding: 20px;
}
.modal-footer {
    display: flex;
    gap: 10px;
    justify-content: flex-end;
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid #e5e7eb;
}
.badge {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 12px;
    font-size: 12px;
    font-weight: 600;
}
.badge-primary { background: #dbeafe; color: #1e40af; }
.badge-warning { background: #fef3c7; color: #92400e; }
.badge-info { background: #ddd6fe; color: #5b21b6; }
.action-buttons {
    display: flex;
    gap: 8px;
}
.btn-icon {
    background: none;
    border: none;
    cursor: pointer;
    font-size: 18px;
    padding: 4px 8px;
    border-radius: 4px;
    transition: background 0.2s;
}
.btn-icon:hover {
    background: #f3f4f6;
}
.btn-icon.btn-danger:hover {
    background: #fee2e2;
}
</style>

<script src="/assets/js/main.js"></script>
<script>
const users = <?= json_encode($users) ?>;

function showCreateUserModal() {
    document.getElementById('createUserModal').style.display = 'flex';
}

function closeModal(modalId) {
    document.getElementById(modalId).style.display = 'none';
}

function editUser(userId) {
    const user = users.find(u => u.id == userId);
    if (!user) return;
    
    document.getElementById('edit_user_id').value = user.id;
    document.getElementById('edit_email').value = user.email;
    document.getElementById('edit_role').value = user.role;
    
    document.getElementById('editUserModal').style.display = 'flex';
}

function deleteUser(userId, username) {
    if (confirm(`Benutzer "${username}" wirklich löschen?\n\nAlle Inhalte dieses Benutzers bleiben erhalten.`)) {
        document.getElementById('delete_user_id').value = userId;
        document.getElementById('deleteUserForm').submit();
    }
}

// Close modal on background click
document.querySelectorAll('.modal').forEach(modal => {
    modal.addEventListener('click', function(e) {
        if (e.target === this) {
            this.style.display = 'none';
        }
    });
});

// Close modal on ESC key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        document.querySelectorAll('.modal').forEach(modal => {
            modal.style.display = 'none';
        });
    }
});
</script>

</body>
</html>