<?php
/**
 * cms-core/public/webview/api/mailchimp-subscribe.php
 * CORE: Mailchimp Newsletter Subscription Handler
 * Processes newsletter signups via Mailchimp API
 * ENHANCED: Added try-catch for error handling
 */

require_once __DIR__ . '/../../../backend/core/init.php';
require_once __DIR__ . '/../../../backend/integrations/integration_helper.php';

header('Content-Type: application/json');

try {
    // Check if Mailchimp is enabled
    if (!IntegrationHelper::isActive('mailchimp')) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Newsletter-Integration ist nicht aktiviert.'
        ]);
        exit;
    }
    
    // Get Mailchimp config
    $apiKey = get_setting('mailchimp_api_key', '');
    $listId = get_setting('mailchimp_list_id', '');
    
    if (empty($apiKey) || empty($listId)) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Mailchimp ist nicht korrekt konfiguriert.'
        ]);
        exit;
    }
    
    // Validate email
    $email = filter_input(INPUT_POST, 'email', FILTER_VALIDATE_EMAIL);
    
    if (!$email) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Ungültige E-Mail-Adresse.'
        ]);
        exit;
    }
    
    // Get datacenter from API key (e.g., us1, us19, etc.)
    $datacenter = substr($apiKey, strpos($apiKey, '-') + 1);
    $url = "https://{$datacenter}.api.mailchimp.com/3.0/lists/{$listId}/members";
    
    // Prepare subscriber data
    $data = [
        'email_address' => $email,
        'status' => 'pending', // Double opt-in for GDPR compliance
        'tags' => ['Website Signup']
    ];
    
    // Make API request
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_USERPWD, 'user:' . $apiKey);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    // Handle cURL errors
    if ($curlError) {
        try {
            Logger::error("Mailchimp cURL error: {$curlError}");
        } catch (Exception $e) {
            error_log('Logger error: ' . $e->getMessage());
        }
        
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Verbindungsfehler. Bitte versuchen Sie es später erneut.'
        ]);
        exit;
    }
    
    // Handle response
    if ($httpCode >= 200 && $httpCode < 300) {
        // Success
        try {
            Logger::info("Mailchimp subscription: {$email}");
        } catch (Exception $e) {
            error_log('Logger error: ' . $e->getMessage());
        }
        
        echo json_encode([
            'success' => true,
            'message' => 'Erfolgreich angemeldet! Bitte bestätigen Sie Ihre E-Mail.'
        ]);
    } else {
        // Error
        $error = json_decode($response, true);
        $errorTitle = $error['title'] ?? 'Anmeldung fehlgeschlagen';
        $errorDetail = $error['detail'] ?? '';
        
        try {
            Logger::error("Mailchimp error for {$email}: {$errorTitle} - {$errorDetail}");
        } catch (Exception $e) {
            error_log('Logger error: ' . $e->getMessage());
        }
        
        // Check for already subscribed
        if ($httpCode === 400 && (
            strpos($errorTitle, 'already') !== false || 
            strpos($errorDetail, 'already') !== false
        )) {
            echo json_encode([
                'success' => false,
                'message' => 'Diese E-Mail-Adresse ist bereits angemeldet.'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Ein Fehler ist aufgetreten. Bitte versuchen Sie es später erneut.'
            ]);
        }
    }
    
} catch (Exception $e) {
    error_log('Mailchimp subscription failed: ' . $e->getMessage());
    
    try {
        Logger::error('Mailchimp subscription error: ' . $e->getMessage());
    } catch (Exception $logError) {
        error_log('Logger error: ' . $logError->getMessage());
    }
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Ein Fehler ist aufgetreten. Bitte versuchen Sie es später erneut.'
    ]);
}