<?php
/** cms-core/public/webview/blog-post.php - COMPLETE with helper functions */

require_once __DIR__ . '/../../backend/core/init.php';

$slug = get_param('slug');

if (!$slug) {
    redirect(blog_url());
}

// Hole Blog-Post
$post = db()->fetchOne(
    "SELECT bp.*, u.username as author_name
     FROM blog_posts bp 
     LEFT JOIN users u ON bp.author_id = u.id 
     WHERE bp.slug = ? AND bp.status = 'published'",
    [$slug]
);

if (!$post) {
    http_response_code(404);
    include __DIR__ . '/404.php';
    exit;
}

// Hole verwandte Posts (3 Stück)
$relatedPosts = db()->fetchAll(
    "SELECT * FROM blog_posts 
     WHERE status = 'published' 
     AND id != ? 
     ORDER BY published_at DESC 
     LIMIT 3",
    [$post['id']]
);

$pageTitle = $post['title'];
$metaDescription = $post['excerpt'] ?? truncate(strip_tags($post['content']), 160);
$bodyClass = 'page-blog-post single-post-modern';

ob_start();
?>

<!-- MODERN BLOG POST DESIGN -->
<article class="modern-blog-post">
    <!-- Hero Section with Image Overlay -->
    <div class="post-hero">
        <?php if ($post['featured_image']): ?>
            <div class="post-hero-image">
                <img src="<?= media_url('blog/' . e($post['featured_image'])) ?>" 
                     alt="<?= e($post['title']) ?>"
                     loading="eager">
                <div class="post-hero-overlay"></div>
            </div>
        <?php endif; ?>
        
        <div class="post-hero-content">
            <div class="container container-narrow">
                <!-- Metadata Pills -->
                <div class="post-meta-pills">
                    <span class="meta-pill">
                        <svg width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                            <path d="M8 3.5a.5.5 0 0 0-1 0V9a.5.5 0 0 0 .252.434l3.5 2a.5.5 0 0 0 .496-.868L8 8.71V3.5z"/>
                            <path d="M8 16A8 8 0 1 0 8 0a8 8 0 0 0 0 16zm7-8A7 7 0 1 1 1 8a7 7 0 0 1 14 0z"/>
                        </svg>
                        <?= format_date($post['published_at'], 'd. M Y') ?>
                    </span>
                    <?php if ($post['author_name']): ?>
                        <span class="meta-pill">
                            <svg width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                                <path d="M11 6a3 3 0 1 1-6 0 3 3 0 0 1 6 0z"/>
                                <path d="M0 8a8 8 0 1 1 16 0A8 8 0 0 1 0 8zm8-7a7 7 0 0 0-5.468 11.37C3.242 11.226 4.805 10 8 10s4.757 1.225 5.468 2.37A7 7 0 0 0 8 1z"/>
                            </svg>
                            <?= e($post['author_name']) ?>
                        </span>
                    <?php endif; ?>
                </div>
                
                <!-- Title -->
                <h1 class="post-hero-title"><?= e($post['title']) ?></h1>
                
                <?php if ($post['excerpt']): ?>
                    <p class="post-hero-excerpt"><?= e($post['excerpt']) ?></p>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Content Section -->
    <div class="post-content-wrapper">
        <div class="container container-narrow">
            <div class="post-content-grid">
                <!-- Main Content -->
                <div class="post-main-content">
                    <div class="post-body prose">
                        <?= nl2br(e($post['content'])) ?>
                    </div>
                    
                    <!-- Share Section -->
                    <div class="post-share-section">
                        <h4>Artikel teilen</h4>
                        <div class="share-buttons">
                            <button onclick="shareOn('twitter')" class="share-btn share-twitter" title="Auf Twitter teilen">
                                <svg width="20" height="20" fill="currentColor" viewBox="0 0 16 16">
                                    <path d="M5.026 15c6.038 0 9.341-5.003 9.341-9.334 0-.14 0-.282-.006-.422A6.685 6.685 0 0 0 16 3.542a6.658 6.658 0 0 1-1.889.518 3.301 3.301 0 0 0 1.447-1.817 6.533 6.533 0 0 1-2.087.793A3.286 3.286 0 0 0 7.875 6.03a9.325 9.325 0 0 1-6.767-3.429 3.289 3.289 0 0 0 1.018 4.382A3.323 3.323 0 0 1 .64 6.575v.045a3.288 3.288 0 0 0 2.632 3.218 3.203 3.203 0 0 1-.865.115 3.23 3.23 0 0 1-.614-.057 3.283 3.283 0 0 0 3.067 2.277A6.588 6.588 0 0 1 .78 13.58a6.32 6.32 0 0 1-.78-.045A9.344 9.344 0 0 0 5.026 15z"/>
                                </svg>
                            </button>
                            <button onclick="shareOn('facebook')" class="share-btn share-facebook" title="Auf Facebook teilen">
                                <svg width="20" height="20" fill="currentColor" viewBox="0 0 16 16">
                                    <path d="M16 8.049c0-4.446-3.582-8.05-8-8.05C3.58 0-.002 3.603-.002 8.05c0 4.017 2.926 7.347 6.75 7.951v-5.625h-2.03V8.05H6.75V6.275c0-2.017 1.195-3.131 3.022-3.131.876 0 1.791.157 1.791.157v1.98h-1.009c-.993 0-1.303.621-1.303 1.258v1.51h2.218l-.354 2.326H9.25V16c3.824-.604 6.75-3.934 6.75-7.951z"/>
                                </svg>
                            </button>
                            <button onclick="shareOn('linkedin')" class="share-btn share-linkedin" title="Auf LinkedIn teilen">
                                <svg width="20" height="20" fill="currentColor" viewBox="0 0 16 16">
                                    <path d="M0 1.146C0 .513.526 0 1.175 0h13.65C15.474 0 16 .513 16 1.146v13.708c0 .633-.526 1.146-1.175 1.146H1.175C.526 16 0 15.487 0 14.854V1.146zm4.943 12.248V6.169H2.542v7.225h2.401zm-1.2-8.212c.837 0 1.358-.554 1.358-1.248-.015-.709-.52-1.248-1.342-1.248-.822 0-1.359.54-1.359 1.248 0 .694.521 1.248 1.327 1.248h.016zm4.908 8.212V9.359c0-.216.016-.432.08-.586.173-.431.568-.878 1.232-.878.869 0 1.216.662 1.216 1.634v3.865h2.401V9.25c0-2.22-1.184-3.252-2.764-3.252-1.274 0-1.845.7-2.165 1.193v.025h-.016a5.54 5.54 0 0 1 .016-.025V6.169h-2.4c.03.678 0 7.225 0 7.225h2.4z"/>
                                </svg>
                            </button>
                            <button onclick="copyLink()" class="share-btn share-link" title="Link kopieren">
                                <svg width="20" height="20" fill="currentColor" viewBox="0 0 16 16">
                                    <path d="M4.715 6.542 3.343 7.914a3 3 0 1 0 4.243 4.243l1.828-1.829A3 3 0 0 0 8.586 5.5L8 6.086a1.002 1.002 0 0 0-.154.199 2 2 0 0 1 .861 3.337L6.88 11.45a2 2 0 1 1-2.83-2.83l.793-.792a4.018 4.018 0 0 1-.128-1.287z"/>
                                    <path d="M6.586 4.672A3 3 0 0 0 7.414 9.5l.775-.776a2 2 0 0 1-.896-3.346L9.12 3.55a2 2 0 1 1 2.83 2.83l-.793.792c.112.42.155.855.128 1.287l1.372-1.372a3 3 0 1 0-4.243-4.243L6.586 4.672z"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                    
                    <!-- Back Button -->
                    <div class="post-navigation">
                        <a href="<?= blog_url() ?>" class="back-to-blog">
                            <svg width="20" height="20" fill="currentColor" viewBox="0 0 16 16">
                                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
                            </svg>
                            Zurück zur Übersicht
                        </a>
                    </div>
                </div>
                
                <!-- Sidebar (on desktop) -->
                <aside class="post-sidebar">
                    <!-- CTA Card or Ad Space -->
                    <div class="cta-card">
                        <h4>📬 Newsletter</h4>
                        <p style="font-size: 14px; color: #6b7280; margin-bottom: 1rem;">
                            Bleiben Sie auf dem Laufenden mit unseren neuesten Artikeln.
                        </p>
                        <a href="<?= page_url('kontakt') ?>" class="btn-cta">
                            Jetzt abonnieren
                        </a>
                    </div>
                </aside>
            </div>
        </div>
    </div>
    
    <!-- Related Posts Section -->
    <?php if (!empty($relatedPosts)): ?>
        <section class="related-posts-section">
            <div class="container">
                <h2 class="section-heading">Weitere interessante Artikel</h2>
                <div class="related-posts-grid">
                    <?php foreach ($relatedPosts as $related): ?>
                        <article class="related-post-card">
                            <?php if ($related['featured_image']): ?>
                                <a href="<?= blog_post_url($related['slug']) ?>" class="related-post-image">
                                    <img src="<?= media_url('blog/' . e($related['featured_image'])) ?>" 
                                         alt="<?= e($related['title']) ?>"
                                         loading="lazy">
                                </a>
                            <?php endif; ?>
                            <div class="related-post-content">
                                <span class="related-post-date">
                                    <?= format_date($related['published_at'], 'd. M Y') ?>
                                </span>
                                <h3 class="related-post-title">
                                    <a href="<?= blog_post_url($related['slug']) ?>">
                                        <?= e($related['title']) ?>
                                    </a>
                                </h3>
                                <?php if ($related['excerpt']): ?>
                                    <p class="related-post-excerpt">
                                        <?= e(truncate($related['excerpt'], 100)) ?>
                                    </p>
                                <?php endif; ?>
                            </div>
                        </article>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
    <?php endif; ?>
</article>

<style>
/* ===============================================
   MODERN BLOG POST STYLES - PROFESSIONAL DESIGN
   =============================================== */

/* Hero Section */
.post-hero {
    position: relative;
    min-height: 70vh;
    display: flex;
    align-items: center;
    justify-content: center;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    overflow: hidden;
}

.post-hero-no-image {
    min-height: 50vh;
}

.post-hero-image {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    z-index: 1;
}

.post-hero-image img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.post-hero-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: linear-gradient(
        180deg,
        rgba(0, 0, 0, 0.3) 0%,
        rgba(0, 0, 0, 0.6) 100%
    );
    z-index: 2;
}

.post-hero-content {
    position: relative;
    z-index: 3;
    padding: 4rem 0;
    width: 100%;
}

.post-meta-pills {
    display: flex;
    gap: 12px;
    margin-bottom: 24px;
    flex-wrap: wrap;
}

.meta-pill {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 8px 16px;
    background: rgba(255, 255, 255, 0.15);
    backdrop-filter: blur(10px);
    border-radius: 20px;
    color: white;
    font-size: 14px;
    font-weight: 500;
    border: 1px solid rgba(255, 255, 255, 0.2);
}

.post-hero-title {
    font-size: 3.5rem;
    font-weight: 800;
    color: white;
    margin-bottom: 24px;
    line-height: 1.2;
    text-shadow: 0 2px 20px rgba(0, 0, 0, 0.3);
}

.post-hero-excerpt {
    font-size: 1.5rem;
    color: rgba(255, 255, 255, 0.95);
    line-height: 1.6;
    max-width: 800px;
    text-shadow: 0 1px 10px rgba(0, 0, 0, 0.2);
}

/* Content Wrapper */
.post-content-wrapper {
    background: #ffffff;
    padding: 4rem 0;
}

.post-content-grid {
    display: grid;
    grid-template-columns: 1fr;
    gap: 3rem;
}

/* Main Content */
.post-main-content {
    max-width: 100%;
}

.post-body.prose {
    font-size: 1.125rem;
    line-height: 1.8;
    color: #374151;
}

.post-body.prose p {
    margin-bottom: 1.5rem;
}

.post-body.prose h2 {
    font-size: 2rem;
    font-weight: 700;
    margin: 3rem 0 1.5rem;
    color: #111827;
}

.post-body.prose h3 {
    font-size: 1.5rem;
    font-weight: 600;
    margin: 2.5rem 0 1rem;
    color: #1f2937;
}

/* Share Section */
.post-share-section {
    margin: 4rem 0;
    padding: 2rem;
    background: linear-gradient(135deg, #f9fafb 0%, #f3f4f6 100%);
    border-radius: 16px;
    border: 1px solid #e5e7eb;
}

.post-share-section h4 {
    font-size: 1.25rem;
    font-weight: 700;
    margin-bottom: 1rem;
    color: #111827;
}

.share-buttons {
    display: flex;
    gap: 12px;
    flex-wrap: wrap;
}

.share-btn {
    width: 48px;
    height: 48px;
    border-radius: 12px;
    border: none;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.3s ease;
    color: white;
}

.share-twitter { background: #1DA1F2; }
.share-facebook { background: #1877F2; }
.share-linkedin { background: #0A66C2; }
.share-link { background: #6b7280; }

.share-btn:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 20px rgba(0, 0, 0, 0.15);
}

/* Navigation */
.post-navigation {
    margin-top: 3rem;
}

.back-to-blog {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: white;
    border: 2px solid #667eea;
    border-radius: 12px;
    color: #667eea;
    font-weight: 600;
    text-decoration: none;
    transition: all 0.3s ease;
}

.back-to-blog:hover {
    background: #667eea;
    color: white;
    transform: translateX(-5px);
}

/* Sidebar */
.post-sidebar {
    display: none;
}

.author-card, .toc-card, .cta-card {
    background: white;
    padding: 2rem;
    border-radius: 16px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    border: 1px solid #e5e7eb;
    margin-bottom: 2rem;
}

.cta-card h4 {
    font-size: 18px;
    font-weight: 700;
    margin-bottom: 0.5rem;
    color: #111827;
}

.btn-cta {
    display: block;
    width: 100%;
    padding: 12px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    text-align: center;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: transform 0.3s ease;
}

.btn-cta:hover {
    transform: translateY(-2px);
}

/* Related Posts */
.related-posts-section {
    padding: 5rem 0;
    background: #f9fafb;
}

.section-heading {
    font-size: 2.5rem;
    font-weight: 800;
    text-align: center;
    margin-bottom: 3rem;
    color: #111827;
}

.related-posts-grid {
    display: grid;
    grid-template-columns: 1fr;
    gap: 2rem;
}

.related-post-card {
    background: white;
    border-radius: 16px;
    overflow: hidden;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    transition: all 0.3s ease;
    border: 1px solid #e5e7eb;
}

.related-post-card:hover {
    transform: translateY(-8px);
    box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
}

.related-post-image {
    display: block;
    width: 100%;
    height: 220px;
    overflow: hidden;
}

.related-post-image img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.3s ease;
}

.related-post-card:hover .related-post-image img {
    transform: scale(1.1);
}

.related-post-content {
    padding: 1.5rem;
}

.related-post-date {
    display: inline-block;
    font-size: 13px;
    color: #6b7280;
    margin-bottom: 8px;
}

.related-post-title {
    font-size: 1.25rem;
    font-weight: 700;
    margin-bottom: 8px;
}

.related-post-title a {
    color: #111827;
    text-decoration: none;
    transition: color 0.3s ease;
}

.related-post-title a:hover {
    color: #667eea;
}

.related-post-excerpt {
    color: #6b7280;
    font-size: 14px;
    line-height: 1.6;
    margin: 0;
}

/* Tablet & Desktop */
@media (min-width: 768px) {
    .post-hero-title {
        font-size: 4rem;
    }
    
    .related-posts-grid {
        grid-template-columns: repeat(3, 1fr);
    }
}

@media (min-width: 1024px) {
    .post-content-grid {
        grid-template-columns: 1fr 300px;
    }
    
    .post-sidebar {
        display: block;
    }
    
    .post-hero-title {
        font-size: 4.5rem;
    }
}

/* Mobile Optimizations */
@media (max-width: 767px) {
    .post-hero {
        min-height: 60vh;
    }
    
    .post-hero-title {
        font-size: 2rem;
    }
    
    .post-hero-excerpt {
        font-size: 1.125rem;
    }
    
    .post-body.prose {
        font-size: 1rem;
    }
}
</style>

<script>
// Share Functions
function shareOn(platform) {
    const url = encodeURIComponent(window.location.href);
    const title = encodeURIComponent(document.title);
    
    let shareUrl = '';
    
    switch(platform) {
        case 'twitter':
            shareUrl = `https://twitter.com/intent/tweet?url=${url}&text=${title}`;
            break;
        case 'facebook':
            shareUrl = `https://www.facebook.com/sharer/sharer.php?u=${url}`;
            break;
        case 'linkedin':
            shareUrl = `https://www.linkedin.com/sharing/share-offsite/?url=${url}`;
            break;
    }
    
    if (shareUrl) {
        window.open(shareUrl, '_blank', 'width=600,height=400');
    }
}

function copyLink() {
    navigator.clipboard.writeText(window.location.href).then(() => {
        alert('Link in die Zwischenablage kopiert! ✅');
    });
}
</script>

<?php
$pageContent = ob_get_clean();
include __DIR__ . '/includes/layout.php';
?>