<?php
/**
 * cms-core/public/webview/contact_form_handler.php
 * Handles contact form submissions from flexible blocks
 * ENHANCED: Added try-catch for error handling
 */

require_once __DIR__ . '/../../backend/core/init.php';
require_once __DIR__ . '/../../backend/core/Mailer.php';

// Only accept POST
if (!is_post()) {
    http_response_code(405);
    die('Method not allowed');
}

// CSRF protection
try {
    CSRF::require();
} catch (Exception $e) {
    set_flash('error', 'Sicherheitsüberprüfung fehlgeschlagen. Bitte versuche es erneut.');
    redirect($_SERVER['HTTP_REFERER'] ?? '/kontakt');
    exit;
}

// Get form data
$name = trim(post('name'));
$email = trim(post('email'));
$phone = trim(post('phone', ''));
$subject = trim(post('subject', 'Kontaktanfrage'));
$message = trim(post('message'));

// Honeypot anti-spam
$honeypot = post('website', ''); // Bots fill this field
if (!empty($honeypot)) {
    // Silently fail for bots
    set_flash('success', 'Vielen Dank für Ihre Nachricht! Wir melden uns bald bei Ihnen.');
    redirect($_SERVER['HTTP_REFERER'] ?? '/kontakt');
    exit;
}

// Validate input
$errors = [];

if (empty($name)) {
    $errors[] = 'Bitte geben Sie Ihren Namen ein.';
}

if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $errors[] = 'Bitte geben Sie eine gültige E-Mail-Adresse ein.';
}

if (empty($message)) {
    $errors[] = 'Bitte geben Sie eine Nachricht ein.';
}

// Rate limiting - max 5 submissions per IP per hour
try {
    $ipKey = 'contact_form_' . md5($_SERVER['REMOTE_ADDR']);
    $submissions = (int)get_setting($ipKey, 0);

    if ($submissions >= 5) {
        $errors[] = 'Zu viele Anfragen. Bitte versuchen Sie es später erneut.';
    }
} catch (Exception $e) {
    error_log('Rate limiting error: ' . $e->getMessage());
    // Continue without rate limiting if it fails
}

// If errors, redirect back
if (!empty($errors)) {
    set_flash('error', implode('<br>', $errors));
    redirect($_SERVER['HTTP_REFERER'] ?? '/kontakt');
    exit;
}

// Prepare email
try {
    $adminEmail = get_setting('contact_email', MAIL_FROM);
    $siteName = get_setting('site_name', 'KynoCanis');
} catch (Exception $e) {
    error_log('Error getting settings: ' . $e->getMessage());
    $adminEmail = MAIL_FROM;
    $siteName = 'Website';
}

// Email body
$emailBody = "
<html>
<head>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 8px 8px 0 0; }
        .content { background: #f9fafb; padding: 20px; border: 1px solid #e5e7eb; }
        .field { margin-bottom: 15px; }
        .field-label { font-weight: bold; color: #667eea; }
        .field-value { margin-top: 5px; padding: 10px; background: white; border-radius: 4px; }
        .footer { padding: 15px; text-align: center; color: #6b7280; font-size: 12px; }
    </style>
</head>
<body>
    <div class='container'>
        <div class='header'>
            <h2>📧 Neue Kontaktanfrage</h2>
            <p style='margin: 0;'>Von: $siteName Website</p>
        </div>
        
        <div class='content'>
            <div class='field'>
                <div class='field-label'>👤 Name:</div>
                <div class='field-value'>" . htmlspecialchars($name) . "</div>
            </div>
            
            <div class='field'>
                <div class='field-label'>📧 E-Mail:</div>
                <div class='field-value'><a href='mailto:" . htmlspecialchars($email) . "'>" . htmlspecialchars($email) . "</a></div>
            </div>";

if (!empty($phone)) {
    $emailBody .= "
            <div class='field'>
                <div class='field-label'>📱 Telefon:</div>
                <div class='field-value'>" . htmlspecialchars($phone) . "</div>
            </div>";
}

$emailBody .= "
            <div class='field'>
                <div class='field-label'>💬 Nachricht:</div>
                <div class='field-value'>" . nl2br(htmlspecialchars($message)) . "</div>
            </div>
            
            <div class='field'>
                <div class='field-label'>🕐 Eingegangen am:</div>
                <div class='field-value'>" . date('d.m.Y H:i:s') . "</div>
            </div>
            
            <div class='field'>
                <div class='field-label'>🌐 IP-Adresse:</div>
                <div class='field-value'>" . $_SERVER['REMOTE_ADDR'] . "</div>
            </div>
        </div>
        
        <div class='footer'>
            <p>Diese E-Mail wurde über das Kontaktformular auf $siteName gesendet.</p>
            <p>Antworten Sie direkt auf diese E-Mail, um den Absender zu kontaktieren.</p>
        </div>
    </div>
</body>
</html>
";

// Plain text version
$plainBody = "Neue Kontaktanfrage von $siteName\n\n";
$plainBody .= "Name: $name\n";
$plainBody .= "E-Mail: $email\n";
if (!empty($phone)) {
    $plainBody .= "Telefon: $phone\n";
}
$plainBody .= "\nNachricht:\n$message\n\n";
$plainBody .= "---\n";
$plainBody .= "Eingegangen am: " . date('d.m.Y H:i:s') . "\n";
$plainBody .= "IP-Adresse: " . $_SERVER['REMOTE_ADDR'] . "\n";

// Send email
try {
    $mailer = new Mailer();
    $result = $mailer->send(
        $adminEmail,                    // To
        $subject,                       // Subject
        $emailBody,                     // HTML body
        $plainBody,                     // Plain body
        $email,                         // Reply-To (sender's email)
        $name                           // Reply-To name
    );
    
    if ($result['success']) {
        // Increment submission counter (expires after 1 hour)
        try {
            set_setting($ipKey, $submissions + 1, 'text');
        } catch (Exception $e) {
            error_log('Failed to update rate limit counter: ' . $e->getMessage());
        }
        
        // Log successful submission
        try {
            Logger::info("Contact form submitted by $name ($email)");
        } catch (Exception $e) {
            error_log('Logger error: ' . $e->getMessage());
        }
        
        // Success message
        set_flash('success', '✅ Vielen Dank für Ihre Nachricht! Wir melden uns schnellstmöglich bei Ihnen.');
        
        // Redirect back
        redirect($_SERVER['HTTP_REFERER'] ?? '/kontakt');
    } else {
        throw new Exception($result['error'] ?? 'Unbekannter Fehler');
    }
    
} catch (Exception $e) {
    error_log('Contact form error: ' . $e->getMessage());
    
    try {
        Logger::error("Contact form error: " . $e->getMessage());
    } catch (Exception $logError) {
        error_log('Logger error: ' . $logError->getMessage());
    }
    
    set_flash('error', 'Fehler beim Senden der Nachricht. Bitte versuchen Sie es später erneut oder kontaktieren Sie uns direkt per E-Mail.');
    redirect($_SERVER['HTTP_REFERER'] ?? '/kontakt');
}