<?php
/**
 * cms-core/public/webview/includes/header.php
 * Site header with navigation - UPDATED with farb_font variables
 */

// Get menu items for header (show_in_menu = 1, with hierarchy)
$headerItems = MenuManager::getHeaderPages();

// Get site settings
$siteName = get_setting('site_name', 'CMS');
$useLogo = get_setting('use_logo', false);
$logoPath = get_setting('site_logo', '');

// Get current slug for active state
$currentSlug = get_current_slug();
$currentUri = $_SERVER['REQUEST_URI'] ?? '/';
?>

<header class="site-header">
    <div class="header-container">
        <!-- Logo / Site Name -->
        <div class="header-logo">
            <?php if ($useLogo && $logoPath): ?>
                <a href="<?= base_url() ?>/">
                    <img src="/cms-core/public/webview/media.php?file=<?= urlencode($logoPath) ?>" 
                         alt="<?= e($siteName) ?>" 
                         class="logo-image">
                </a>
            <?php else: ?>
                <a href="<?= base_url() ?>/" class="logo-text">
                    <?= e($siteName) ?>
                </a>
            <?php endif; ?>
        </div>
        
        <!-- Mobile Hamburger -->
        <button class="mobile-menu-toggle" onclick="toggleMobileMenu()" aria-label="Menu">
            <span></span>
            <span></span>
            <span></span>
        </button>
        
        <!-- Navigation -->
        <nav class="header-nav">
            <ul class="nav-menu">
                <?php foreach ($headerItems as $item): ?>
                    <?php 
                    $hasChildren = !empty($item['children']);
                    
                    // Check if this item or any of its children are active
                    $isActive = ($currentSlug === $item['slug']) || 
                                ($currentUri === '/' && $item['slug'] === 'index');
                    
                    // Check if any child is active
                    $childActive = false;
                    if ($hasChildren) {
                        foreach ($item['children'] as $child) {
                            if ($currentSlug === $child['slug']) {
                                $childActive = true;
                                $isActive = true;
                                break;
                            }
                        }
                    }
                    
                    // Use menu_label if set, otherwise use title
                    $displayLabel = !empty($item['menu_label']) ? $item['menu_label'] : $item['title'];
                    
                    // Use the URL generated by MenuManager
                    $itemUrl = $item['url'] ?? page_url($item['slug']);
                    ?>
                    
                    <li class="nav-item <?= $hasChildren ? 'has-dropdown' : '' ?> <?= $isActive ? 'active' : '' ?>">
                        <a href="<?= e($itemUrl) ?>" class="nav-link">
                            <?= e($displayLabel) ?>
                            <?php if ($hasChildren): ?>
                                <svg class="dropdown-arrow" width="12" height="12" viewBox="0 0 12 12" fill="currentColor">
                                    <path d="M6 8L2 4h8L6 8z"/>
                                </svg>
                            <?php endif; ?>
                        </a>
                        
                        <?php if ($hasChildren): ?>
                            <ul class="dropdown-menu">
                                <?php foreach ($item['children'] as $child): ?>
                                    <?php 
                                    // Use menu_label if set, otherwise use title
                                    $childLabel = !empty($child['menu_label']) ? $child['menu_label'] : $child['title'];
                                    $childUrl = $child['url'] ?? page_url($child['slug']);
                                    $childIsActive = ($currentSlug === $child['slug']);
                                    ?>
                                    
                                    <li class="dropdown-item <?= $childIsActive ? 'active' : '' ?>">
                                        <a href="<?= e($childUrl) ?>" class="dropdown-link">
                                            <?= e($childLabel) ?>
                                        </a>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>
                    </li>
                <?php endforeach; ?>
            </ul>
        </nav>
    </div>
</header>

<style>
/* ===========================
   HEADER STYLES - Uses farb_font variables
   =========================== */

.site-header {
    background: var(--global-header-bg-color, #ffffff);
    color: var(--global-header-text-color, #333333);
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    position: sticky;
    top: 0;
    z-index: 1000;
}

.header-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
    min-height: 70px;
}

.header-logo .logo-image {
    max-height: 50px;
    width: auto;
}

/* Logo/Sitename - NO GRADIENT, uses header_heading_color from farb_font */
.header-logo .logo-text {
    font-size: 24px;
    font-weight: 700;
    text-decoration: none;
    /* Color is set by farb_font.php via --global-header-heading-color */
    /* DO NOT set color here - let farb_font.php handle it */
}

.header-logo .logo-text:hover {
    opacity: 0.8;
}

/* Mobile Toggle */
.mobile-menu-toggle {
    display: none;
    flex-direction: column;
    gap: 4px;
    background: none;
    border: none;
    cursor: pointer;
    padding: 8px;
}

.mobile-menu-toggle span {
    width: 25px;
    height: 3px;
    background: var(--global-header-text-color, #333);
    border-radius: 2px;
    transition: all 0.3s;
}

.mobile-menu-toggle.active span:nth-child(1) {
    transform: rotate(45deg) translateY(8px);
}

.mobile-menu-toggle.active span:nth-child(2) {
    opacity: 0;
}

.mobile-menu-toggle.active span:nth-child(3) {
    transform: rotate(-45deg) translateY(-8px);
}

/* Navigation */
.header-nav {
    flex: 1;
    display: flex;
    justify-content: flex-end;
}

.nav-menu {
    display: flex;
    list-style: none;
    margin: 0;
    padding: 0;
    gap: 8px;
    align-items: flex-start;
}

.nav-item {
    position: relative;
}

/* Menu Links use header_link_color */
.nav-link {
    display: flex;
    align-items: center;
    gap: 4px;
    padding: 12px 16px;
    color: var(--global-header-link-color, #4b5563);
    text-decoration: none;
    font-weight: 500;
    border-radius: 6px;
    transition: all 0.2s;
}

/* Hover/Active uses header_link_hover_color */
.nav-link:hover,
.nav-item.active .nav-link {
    background: rgba(0,0,0,0.05);
    color: var(--global-header-link-hover-color, #667eea);
}

.dropdown-arrow {
    transition: transform 0.3s;
}

/* Dropdown Menu - DESKTOP */
.dropdown-menu {
    position: absolute;
    top: 100%;
    left: 0;
    min-width: 200px;
    background: var(--global-header-bg-color, white);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    border-radius: 8px;
    padding: 8px 0;
    margin-top: 4px;
    opacity: 0;
    visibility: hidden;
    transform: translateY(-10px);
    transition: all 0.3s;
    list-style: none;
    z-index: 100;
}

/* Show dropdown on hover - DESKTOP */
.nav-item.has-dropdown:hover .dropdown-menu {
    opacity: 1;
    visibility: visible;
    transform: translateY(0);
}

.nav-item.has-dropdown:hover .dropdown-arrow {
    transform: rotate(180deg);
}

.dropdown-item {
    list-style: none;
}

.dropdown-item.active .dropdown-link {
    background: rgba(0,0,0,0.05);
    color: var(--global-header-link-hover-color, #667eea);
}

/* Dropdown links also use header_link_color */
.dropdown-link {
    display: block;
    padding: 10px 20px;
    color: var(--global-header-link-color, #4b5563);
    text-decoration: none;
    transition: all 0.2s;
}

.dropdown-link:hover {
    background: rgba(0,0,0,0.05);
    color: var(--global-header-link-hover-color, #667eea);
}

/* ===========================
   MOBILE STYLES
   =========================== */

@media (max-width: 768px) {
    .mobile-menu-toggle {
        display: flex;
    }
    
    .header-nav {
        position: fixed;
        top: 70px;
        left: 0;
        right: 0;
        background: var(--global-header-bg-color, #ffffff);
        max-height: 0;
        overflow: hidden;
        transition: max-height 0.3s ease;
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
    }
    
    .header-nav.active {
        max-height: calc(100vh - 70px);
        overflow-y: auto;
    }
    
    .nav-menu {
        flex-direction: column;
        width: 100%;
        gap: 0;
        padding: 12px 0;
        align-items: stretch;
    }
    
    .nav-item {
        width: 100%;
    }
    
    .nav-link {
        padding: 16px 20px;
        border-radius: 0;
        justify-content: space-between;
    }
    
    /* Mobile Dropdown - Click to toggle */
    .dropdown-menu {
        position: static;
        opacity: 1;
        visibility: visible;
        transform: none;
        box-shadow: none;
        background: rgba(0,0,0,0.02);
        margin: 0;
        max-height: 0;
        overflow: hidden;
        transition: max-height 0.3s ease;
    }
    
    .nav-item.has-dropdown.dropdown-open .dropdown-menu {
        max-height: 500px;
    }
    
    .nav-item.has-dropdown.dropdown-open .dropdown-arrow {
        transform: rotate(180deg);
    }
    
    .dropdown-link {
        padding: 12px 20px 12px 40px;
        font-size: 14px;
    }
}
</style>

<script>
// Toggle mobile menu
function toggleMobileMenu() {
    const nav = document.querySelector('.header-nav');
    const toggle = document.querySelector('.mobile-menu-toggle');
    
    nav.classList.toggle('active');
    toggle.classList.toggle('active');
}

// Handle dropdown clicks on mobile
document.addEventListener('DOMContentLoaded', function() {
    const dropdownItems = document.querySelectorAll('.nav-item.has-dropdown');
    
    dropdownItems.forEach(item => {
        const link = item.querySelector('.nav-link');
        
        // On mobile, prevent navigation and toggle dropdown
        link.addEventListener('click', function(e) {
            if (window.innerWidth <= 768) {
                e.preventDefault();
                item.classList.toggle('dropdown-open');
                
                // Close other dropdowns
                dropdownItems.forEach(otherItem => {
                    if (otherItem !== item) {
                        otherItem.classList.remove('dropdown-open');
                    }
                });
            }
        });
    });
    
    // Close mobile menu when clicking outside
    document.addEventListener('click', function(e) {
        const nav = document.querySelector('.header-nav');
        const toggle = document.querySelector('.mobile-menu-toggle');
        
        if (nav.classList.contains('active') && 
            !nav.contains(e.target) && 
            !toggle.contains(e.target)) {
            nav.classList.remove('active');
            toggle.classList.remove('active');
        }
    });
});
</script>