<?php
/**
 * cms-core/public/webview/index.php
 * Homepage - loads page with slug "index" from database
 * FIX: Don't manually require FlexibleBlocks - init.php handles it
 */

require_once __DIR__ . '/../../backend/core/init.php';

// Always load the "index" page
$pageSlug = 'index';

// Fetch page with COALESCE for page_icon
$page = db()->fetchOne(
    "SELECT id, slug, title, COALESCE(page_icon, '🏠') AS page_icon, meta_description, 
            status, show_in_menu, show_in_footer, is_legal_page, is_system_page, 
            is_deletable, menu_label, menu_order, page_type, uses_flexible_blocks, parent_id, content
     FROM pages 
     WHERE slug = ?", 
    [$pageSlug]
);

// If index page doesn't exist, create it
if (!$page) {
    error_log("Index page not found - creating...");
    
    try {
        $pageId = db()->insert(
            "INSERT INTO pages (slug, title, page_icon, meta_description, status, show_in_menu, menu_label, menu_order, is_deletable, is_system_page, page_type, uses_flexible_blocks) 
             VALUES ('index', 'Home', '🏠', 'Startseite', 'published', 1, 'Home', 0, 0, 1, 'flexible', 1)"
        );
        
        $page = db()->fetchOne("SELECT * FROM pages WHERE id = ?", [$pageId]);
        error_log("Index page created with ID: {$pageId}");
    } catch (Exception $e) {
        error_log("Failed to create index page: " . $e->getMessage());
        http_response_code(500);
        die("Homepage konnte nicht geladen werden.");
    }
}

// Load blocks (ONLY if FlexibleBlocks class exists)
$blocks = [];
if ($page['uses_flexible_blocks'] == 1 && class_exists('FlexibleBlocks')) {
    try {
        $blocks = FlexibleBlocks::getPageBlocks($pageSlug);
    } catch (Exception $e) {
        error_log("Error loading blocks for index: " . $e->getMessage());
    }
}

// Set page meta
$pageTitle = $page['title'];
$metaDescription = $page['meta_description'] ?? '';
$bodyClass = 'page-home';

// Check for custom template in customizations
$customTemplate = APP_ROOT . '/customizations/templates/homepage.php';
if (file_exists($customTemplate)) {
    include $customTemplate;
    exit;
}

// Render blocks
ob_start();

if (!empty($blocks) && class_exists('FlexibleBlocks')) {
    foreach ($blocks as $block) {
        try {
            echo FlexibleBlocks::renderBlock($block);
        } catch (Exception $e) {
            error_log("Error rendering block: " . $e->getMessage());
        }
    }
} else {
    // Fallback if no blocks or FlexibleBlocks not loaded
    ?>
    <section class="hero-section" style="padding: 100px 20px; text-align: center; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white;">
        <div class="container" style="max-width: 1200px; margin: 0 auto;">
            <h1 style="font-size: 48px; margin-bottom: 20px;"><?= e($page['title']) ?></h1>
            <?php if (!empty($page['content'])): ?>
                <?= $page['content'] ?>
            <?php else: ?>
                <p style="font-size: 20px;">Willkommen auf unserer Website!</p>
                <p style="margin-top: 20px;">
                    <a href="/admin/page_flexible_edit.php?slug=index" style="background: white; color: #667eea; padding: 15px 30px; border-radius: 8px; text-decoration: none; display: inline-block;">
                        Inhalt bearbeiten
                    </a>
                </p>
            <?php endif; ?>
        </div>
    </section>
    <?php
}

$pageContent = ob_get_clean();

// Include main layout
include __DIR__ . '/includes/layout.php';