<?php
/**
 * database_installer.php
 * COMPLETE DATABASE SCHEMA - All production tables
 * Updated: 2025 - Includes ALL necessary tables + Color Experiment
 */

class DatabaseInstaller {
    
    private $pdo;
    private $errors = [];
    private $log = [];
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Install complete database schema
     */
    public function install() {
        try {
            // Create tables in correct order (respecting foreign keys)
            $this->createUsersTable();
            $this->createPagesTable();
            $this->createSettingsTable();
            $this->createSessionsTable();
            $this->createPasswordResetsTable();
            $this->createLoginAttemptsTable();
            $this->createPageBlocksTable();
            $this->createFlexibleBlocksTable();
            $this->createBlogPostsTable();
            $this->createCustomGlobalStylesTable();
            $this->createAnalyticsSessionsTable();
            $this->createAnalyticsPageviewsTable();
            $this->createMediaTable();
            
            // Insert default data
            $this->insertDefaultSettings();
            
            return [
                'success' => true,
                'log' => $this->log,
                'errors' => $this->errors
            ];
            
        } catch (Exception $e) {
            $this->errors[] = $e->getMessage();
            return [
                'success' => false,
                'log' => $this->log,
                'errors' => $this->errors
            ];
        }
    }
    
    /**
     * Create users table
     */
    private function createUsersTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `users` (
            `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `username` VARCHAR(50) NOT NULL,
            `email` VARCHAR(255) NOT NULL,
            `password` VARCHAR(255) NOT NULL COMMENT 'For legacy compatibility',
            `password_hash` VARCHAR(255) NOT NULL,
            `role` ENUM('super_admin', 'admin', 'editor', 'viewer') DEFAULT 'viewer',
            `created_at` DATETIME NOT NULL,
            `last_login` DATETIME NULL,
            `is_active` TINYINT(1) DEFAULT 1,
            PRIMARY KEY (`id`),
            UNIQUE KEY `username` (`username`),
            UNIQUE KEY `email` (`email`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Users table created';
    }
    
    /**
     * Create pages table
     */
    private function createPagesTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `pages` (
            `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `slug` VARCHAR(255) NOT NULL,
            `page_icon` VARCHAR(20) DEFAULT '📄',
            `title` VARCHAR(255) NOT NULL,
            `meta_title` VARCHAR(255) NULL,
            `meta_description` TEXT NULL,
            `content` LONGTEXT NULL,
            `blocks_json` LONGTEXT NULL,
            `hero_image` VARCHAR(500) NULL,
            `last_modified` DATETIME NOT NULL,
            `created_at` DATETIME NOT NULL,
            `status` ENUM('draft', 'published') DEFAULT 'draft',
            `uses_flexible_blocks` TINYINT(1) DEFAULT 1,
            `show_in_menu` TINYINT(1) DEFAULT 1,
            `show_in_header` TINYINT(1) DEFAULT 1 COMMENT 'Im Header anzeigen?',
            `menu_label` VARCHAR(100) NULL,
            `menu_order` INT DEFAULT 0,
            `is_deletable` TINYINT(1) DEFAULT 1,
            `page_type` VARCHAR(50) DEFAULT 'flexible',
            `parent_id` INT(11) UNSIGNED NULL,
            `is_legal_page` TINYINT(1) DEFAULT 0 COMMENT 'Ist dies eine rechtliche Pflichtseite?',
            `legal_page_type` VARCHAR(50) NULL COMMENT 'Art: impressum, datenschutz, agb, cookies',
            `show_in_footer` TINYINT(1) DEFAULT 0 COMMENT 'Im Footer anzeigen?',
            `is_system_page` TINYINT(1) DEFAULT 0 COMMENT 'System-Seite (nicht löschbar)',
            `author_id` INT(11) UNSIGNED NULL,
            PRIMARY KEY (`id`),
            UNIQUE KEY `slug` (`slug`),
            KEY `status` (`status`),
            KEY `parent_id` (`parent_id`),
            KEY `menu_order` (`menu_order`),
            KEY `author_id` (`author_id`),
            FOREIGN KEY (`parent_id`) REFERENCES `pages`(`id`) ON DELETE SET NULL,
            FOREIGN KEY (`author_id`) REFERENCES `users`(`id`) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Pages table created';
    }
    
    /**
     * Create settings table
     */
    private function createSettingsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `settings` (
            `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `setting_key` VARCHAR(100) NOT NULL,
            `setting_value` LONGTEXT,
            `setting_type` VARCHAR(50) DEFAULT 'text',
            `description` TEXT NULL,
            `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `setting_key` (`setting_key`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Settings table created';
    }
    
    /**
     * Create sessions table
     */
    private function createSessionsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `sessions` (
            `id` VARCHAR(128) NOT NULL,
            `user_id` INT(11) UNSIGNED NULL,
            `ip_address` VARCHAR(45) NULL,
            `user_agent` TEXT NULL,
            `last_activity` DATETIME NOT NULL,
            `data` LONGTEXT NULL,
            PRIMARY KEY (`id`),
            KEY `user_id` (`user_id`),
            KEY `last_activity` (`last_activity`),
            FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Sessions table created';
    }
    
    /**
     * Create password_resets table
     */
    private function createPasswordResetsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `password_resets` (
            `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `user_id` INT(11) UNSIGNED NOT NULL,
            `token` VARCHAR(255) NOT NULL,
            `expires_at` DATETIME NOT NULL,
            `used` TINYINT(1) DEFAULT 0,
            `created_at` DATETIME NOT NULL,
            PRIMARY KEY (`id`),
            UNIQUE KEY `token` (`token`),
            KEY `user_id` (`user_id`),
            KEY `expires_at` (`expires_at`),
            FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Password resets table created';
    }
    
    /**
     * Create login_attempts table
     */
    private function createLoginAttemptsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `login_attempts` (
            `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `ip_address` VARCHAR(45) NOT NULL,
            `attempt_time` DATETIME NOT NULL,
            `username` VARCHAR(100) NULL,
            `successful` TINYINT(1) DEFAULT 0,
            PRIMARY KEY (`id`),
            KEY `ip_address` (`ip_address`),
            KEY `attempt_time` (`attempt_time`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Login attempts table created';
    }
    
    /**
     * Create page_blocks table
     */
    private function createPageBlocksTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `page_blocks` (
            `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `page_slug` VARCHAR(255) NOT NULL,
            `block_key` VARCHAR(100) NULL,
            `block_title` VARCHAR(255) NULL,
            `block_type` VARCHAR(50) NOT NULL,
            `block_type_template` VARCHAR(100) NULL,
            `block_value` LONGTEXT NULL,
            `block_data` LONGTEXT NULL,
            `sort_order` INT DEFAULT 0,
            `created_at` DATETIME NOT NULL,
            `updated_at` DATETIME NOT NULL,
            PRIMARY KEY (`id`),
            KEY `page_slug` (`page_slug`),
            KEY `sort_order` (`sort_order`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Page blocks table created';
    }
    
    /**
     * Create flexible_blocks table (legacy support)
     */
    private function createFlexibleBlocksTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `flexible_blocks` (
            `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `page_slug` VARCHAR(255) NOT NULL,
            `block_type` VARCHAR(50) NOT NULL,
            `content` LONGTEXT NULL,
            `display_order` INT DEFAULT 0,
            `created_at` DATETIME NOT NULL,
            `updated_at` DATETIME NOT NULL,
            PRIMARY KEY (`id`),
            KEY `page_slug` (`page_slug`),
            KEY `display_order` (`display_order`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Flexible blocks table created';
    }
    
    /**
     * Create blog_posts table
     */
    private function createBlogPostsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `blog_posts` (
            `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `title` VARCHAR(255) NOT NULL,
            `slug` VARCHAR(255) NOT NULL,
            `excerpt` TEXT NULL,
            `content` LONGTEXT NULL,
            `featured_image` VARCHAR(500) NULL,
            `author_id` INT(11) UNSIGNED NULL,
            `status` ENUM('draft', 'published') DEFAULT 'draft',
            `created_at` DATETIME NOT NULL,
            `updated_at` DATETIME NOT NULL,
            `published_at` DATETIME NULL,
            `views` INT DEFAULT 0,
            `meta_title` VARCHAR(255) NULL,
            `meta_description` TEXT NULL,
            PRIMARY KEY (`id`),
            UNIQUE KEY `slug` (`slug`),
            KEY `status` (`status`),
            KEY `author_id` (`author_id`),
            KEY `published_at` (`published_at`),
            FOREIGN KEY (`author_id`) REFERENCES `users`(`id`) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Blog posts table created';
    }
    
    /**
     * Create custom_global_styles table
     */
    private function createCustomGlobalStylesTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `custom_global_styles` (
            `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `setting_key` VARCHAR(100) NOT NULL,
            `setting_value` LONGTEXT NULL,
            `setting_type` VARCHAR(50) DEFAULT 'text',
            `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `setting_key` (`setting_key`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Custom global styles table created';
    }
    
    /**
     * Create analytics_sessions table
     */
    private function createAnalyticsSessionsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `analytics_sessions` (
            `id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            `session_hash` VARCHAR(64) NOT NULL,
            `visitor_hash` VARCHAR(64) NOT NULL,
            `started_at` DATETIME NOT NULL,
            `last_activity` DATETIME NOT NULL,
            `duration` INT DEFAULT 0 COMMENT 'Duration in seconds',
            `pageviews` INT DEFAULT 0,
            `bounced` TINYINT(1) DEFAULT 0 COMMENT '1 if only one page viewed',
            `entry_page` VARCHAR(500) NULL,
            `exit_page` VARCHAR(500) NULL,
            `created_at` DATETIME NOT NULL,
            PRIMARY KEY (`id`),
            UNIQUE KEY `session_hash` (`session_hash`),
            KEY `visitor_hash` (`visitor_hash`),
            KEY `started_at` (`started_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Analytics sessions table created';
    }
    
    /**
     * Create analytics_pageviews table
     */
    private function createAnalyticsPageviewsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `analytics_pageviews` (
            `id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            `page_url` VARCHAR(500) NOT NULL,
            `page_title` VARCHAR(255) NULL,
            `visitor_hash` VARCHAR(64) NOT NULL COMMENT 'Anonymized visitor identifier',
            `session_hash` VARCHAR(64) NOT NULL COMMENT 'Session identifier',
            `is_new_visitor` TINYINT(1) DEFAULT 0,
            `is_returning` TINYINT(1) DEFAULT 0,
            `device_type` VARCHAR(20) NULL COMMENT 'mobile, tablet, desktop',
            `country` VARCHAR(10) NULL COMMENT 'ISO country code',
            `country_name` VARCHAR(100) NULL,
            `city` VARCHAR(100) NULL,
            `referrer` VARCHAR(500) NULL,
            `viewed_at` DATETIME NOT NULL,
            `created_at` DATETIME NOT NULL,
            PRIMARY KEY (`id`),
            KEY `page_url` (`page_url`(255)),
            KEY `visitor_hash` (`visitor_hash`),
            KEY `session_hash` (`session_hash`),
            KEY `viewed_at` (`viewed_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Analytics pageviews table created';
    }
    
    /**
     * Create media table
     */
    private function createMediaTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `media` (
            `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `filename` VARCHAR(255) NOT NULL,
            `original_name` VARCHAR(255) NOT NULL,
            `file_path` VARCHAR(500) NOT NULL,
            `file_type` VARCHAR(100) NULL,
            `file_size` INT NULL,
            `mime_type` VARCHAR(100) NULL,
            `width` INT NULL,
            `height` INT NULL,
            `uploaded_by` INT(11) UNSIGNED NULL,
            `uploaded_at` DATETIME NOT NULL,
            `alt_text` VARCHAR(255) NULL,
            `title` VARCHAR(255) NULL,
            PRIMARY KEY (`id`),
            KEY `uploaded_by` (`uploaded_by`),
            KEY `file_type` (`file_type`),
            FOREIGN KEY (`uploaded_by`) REFERENCES `users`(`id`) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Media table created';
    }
    
    /**
     * Create color_settings table (Color Experiment Feature)
     */
    private function createColorSettingsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `color_settings` (
            `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `setting_key` VARCHAR(100) NOT NULL,
            `setting_value` TEXT NULL,
            `setting_type` VARCHAR(50) DEFAULT 'text',
            `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `setting_key` (`setting_key`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Color settings table created';
    }
    
    /**
     * Create color_experiments table (Color Experiment History)
     */
    private function createColorExperimentsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `color_experiments` (
            `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `experiment_name` VARCHAR(255) NOT NULL,
            `color_data` LONGTEXT NOT NULL COMMENT 'JSON with all colors',
            `created_at` DATETIME NOT NULL,
            `created_by` INT(11) UNSIGNED NULL,
            `is_active` TINYINT(1) DEFAULT 0,
            `notes` TEXT NULL,
            PRIMARY KEY (`id`),
            KEY `created_by` (`created_by`),
            KEY `is_active` (`is_active`),
            FOREIGN KEY (`created_by`) REFERENCES `users`(`id`) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
        $this->log[] = '✅ Color experiments table created';
    }
    
    /**
     * Insert default settings
     */
    private function insertDefaultSettings() {
        $defaultSettings = [
            ['site_name', 'Meine Website', 'text', 'Name der Website'],
            ['site_email', 'info@example.com', 'text', 'Kontakt-Email'],
            ['blog_enabled', '0', 'boolean', 'Blog-Funktion aktivieren'],
            ['items_per_page', '10', 'number', 'Einträge pro Seite'],
            ['maintenance_mode', '0', 'boolean', 'Wartungsmodus'],
            ['primary_color', '#667eea', 'color', 'Hauptfarbe'],
            ['secondary_color', '#764ba2', 'color', 'Sekundärfarbe'],
            ['accent_color', '#10b981', 'color', 'Akzentfarbe'],
            ['font_heading', 'Poppins', 'text', 'Schriftart Überschriften'],
            ['font_body', 'Inter', 'text', 'Schriftart Fließtext'],
            ['logo_url', '', 'text', 'Logo URL'],
            ['favicon_url', '', 'text', 'Favicon URL'],
            ['analytics_enabled', '0', 'boolean', 'Analytics aktivieren'],
            ['google_analytics_id', '', 'text', 'Google Analytics ID'],
            ['cloudflare_cdn_enabled', '0', 'boolean', 'Cloudflare CDN aktivieren'],
            ['cloudflare_cdn_url', '', 'text', 'Cloudflare CDN URL'],
            ['cms_version', '1.0.0', 'text', 'CMS Version'],
            ['cms_update_available', '0', 'boolean', 'Update verfügbar'],
            ['cms_update_version', '', 'text', 'Neueste Version']
        ];
        
        $stmt = $this->pdo->prepare(
            "INSERT IGNORE INTO settings (setting_key, setting_value, setting_type, description) VALUES (?, ?, ?, ?)"
        );
        
        foreach ($defaultSettings as $setting) {
            $stmt->execute($setting);
        }
        
        $this->log[] = '✅ Default settings inserted';
    }
    
    /**
     * Check if database is already installed
     */
    public function isInstalled() {
        try {
            $result = $this->pdo->query("SHOW TABLES LIKE 'users'");
            return $result->rowCount() > 0;
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Get list of existing tables
     */
    public function getExistingTables() {
        try {
            $result = $this->pdo->query("SHOW TABLES");
            return $result->fetchAll(PDO::FETCH_COLUMN);
        } catch (Exception $e) {
            return [];
        }
    }
}