<?php
/**
 * install-cms.php - ManiCMS Installation Script
 * VERSION: 3.4 - With License as Step 1
 * 
 * Steps:
 * 0. Language Selection (DE/EN)
 * 1. License Key (Optional)
 * 2. Database Configuration
 * 3. Website Configuration
 * 4. Final Installation
 */

// Prevent re-installation
if (file_exists(__DIR__ . '/.env')) {
    die('❌ CMS already installed. Delete .env file to reinstall.<br><a href="/">Go to homepage</a>');
}

session_start();

// Get selected language (default: German)
$installLang = $_SESSION['install_lang'] ?? 'de';

// Simple translation function for install process
function ti($de, $en) {
    global $installLang;
    return $installLang === 'en' ? $en : $de;
}

$step = isset($_GET['step']) ? (int)$_GET['step'] : 0;
$errors = [];
$success = false;
$updateLog = [];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    
    // Step 0: Language Selection
    if ($step === 0) {
        $lang = $_POST['language'] ?? 'de';
        if (in_array($lang, ['de', 'en'])) {
            $_SESSION['install_lang'] = $lang;
            $installLang = $lang;
        }
        header('Location: install-cms.php?step=1');
        exit;
    }
    
    // Step 1: License Key (Optional)
    if ($step === 1) {
        $licenseKey = trim($_POST['license_key'] ?? '');
        $skipLicense = isset($_POST['skip_license']);
        
        $licenseValid = false;
        $licenseData = null;
        
        if (!empty($licenseKey)) {
            // Validate license format
            if (!preg_match('/^MANI-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/', $licenseKey)) {
                $errors[] = ti('Ungültiges Lizenz-Format. Erwartet: MANI-XXXX-XXXX-XXXX-XXXX', 'Invalid license format. Expected: MANI-XXXX-XXXX-XXXX-XXXX');
            } else {
                // Verify license with manidesign.de API
                $apiUrl = 'https://manidesign.de/wp-json/manicms/v1/verify-license';
                
                $ch = curl_init($apiUrl);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
                    'license_key' => $licenseKey,
                    'domain' => $_SERVER['HTTP_HOST'] ?? 'unknown'
                ]));
                curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
                curl_setopt($ch, CURLOPT_TIMEOUT, 10);
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                
                if ($httpCode === 200 && $response) {
                    $result = json_decode($response, true);
                    if ($result && $result['valid'] === true) {
                        $licenseValid = true;
                        $licenseData = $result;
                    } else {
                        $errors[] = ti('Lizenz-Validierung fehlgeschlagen: ', 'License validation failed: ') . ($result['message'] ?? ti('Unbekannter Fehler', 'Unknown error'));
                    }
                } else {
                    // License API nicht erreichbar - trotzdem weiter
                    $errors[] = ti('Lizenz-Server nicht erreichbar. Du kannst die Lizenz später in den Einstellungen aktivieren.', 'License server not reachable. You can activate the license later in settings.');
                    $licenseKey = ''; // Reset license key
                }
            }
        }
        
        if (empty($errors)) {
            $_SESSION['install_license'] = [
                'key' => $licenseKey,
                'valid' => $licenseValid,
                'data' => $licenseData
            ];
            
            header('Location: install-cms.php?step=2');
            exit;
        }
    }
    
    // Step 2: Database Configuration
    elseif ($step === 2) {
        $dbHost = trim($_POST['db_host'] ?? '');
        $dbName = trim($_POST['db_name'] ?? '');
        $dbUser = trim($_POST['db_user'] ?? '');
        $dbPass = $_POST['db_pass'] ?? '';
        
        if (empty($dbHost)) $errors[] = ti('Datenbank-Host fehlt', 'Database host required');
        if (empty($dbName)) $errors[] = ti('Datenbank-Name fehlt', 'Database name required');
        if (empty($dbUser)) $errors[] = ti('Datenbank-Benutzer fehlt', 'Database user required');
        
        if (empty($errors)) {
            try {
                $dsn = "mysql:host={$dbHost};dbname={$dbName};charset=utf8mb4";
                $pdo = new PDO($dsn, $dbUser, $dbPass, [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
                ]);
                
                $_SESSION['install_db'] = [
                    'host' => $dbHost,
                    'name' => $dbName,
                    'user' => $dbUser,
                    'pass' => $dbPass
                ];
                
                header('Location: install-cms.php?step=3');
                exit;
                
            } catch (PDOException $e) {
                $errors[] = ti('Datenbankverbindung fehlgeschlagen: ', 'Database connection failed: ') . $e->getMessage();
            }
        }
    }
    
    // Step 3: Site Configuration
    elseif ($step === 3) {
        $siteUrl = trim($_POST['site_url'] ?? '');
        $siteName = trim($_POST['site_name'] ?? '');
        $adminEmail = trim($_POST['admin_email'] ?? '');
        $adminUser = trim($_POST['admin_user'] ?? '');
        $adminPass = $_POST['admin_pass'] ?? '';
        $adminPassConfirm = $_POST['admin_pass_confirm'] ?? '';
        $forceSsl = isset($_POST['force_ssl']) && $_POST['force_ssl'] === '1';
        
        if (empty($siteUrl)) $errors[] = ti('Website-URL fehlt', 'Website URL required');
        if (empty($siteName)) $errors[] = ti('Website-Name fehlt', 'Website name required');
        if (empty($adminEmail) || !filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
            $errors[] = ti('Gültige Admin-Email fehlt', 'Valid admin email required');
        }
        if (empty($adminUser)) $errors[] = ti('Admin-Benutzername fehlt', 'Admin username required');
        if (empty($adminPass)) $errors[] = ti('Admin-Passwort fehlt', 'Admin password required');
        if ($adminPass !== $adminPassConfirm) $errors[] = ti('Passwörter stimmen nicht überein', 'Passwords do not match');
        if (strlen($adminPass) < 8) $errors[] = ti('Passwort muss mindestens 8 Zeichen lang sein', 'Password must be at least 8 characters');
        
        if (!preg_match('#^https?://#', $siteUrl)) {
            $protocol = $forceSsl ? 'https://' : 'http://';
            $siteUrl = $protocol . $siteUrl;
        }
        
        $siteUrl = rtrim($siteUrl, '/');
        
        if (empty($errors)) {
            $_SESSION['install_site'] = [
                'url' => $siteUrl,
                'name' => $siteName,
                'email' => $adminEmail,
                'user' => $adminUser,
                'pass' => $adminPass,
                'force_ssl' => $forceSsl
            ];
            
            header('Location: install-cms.php?step=4');
            exit;
        }
    }
    
    // Step 4: Final Installation
    elseif ($step === 4) {
        $db = $_SESSION['install_db'] ?? null;
        $site = $_SESSION['install_site'] ?? null;
        $license = $_SESSION['install_license'] ?? ['key' => '', 'valid' => false, 'data' => null];
        
        if (!$db || !$site) {
            $errors[] = ti('Session abgelaufen. Bitte starte von vorne.', 'Session expired. Please start over.');
        } else {
            try {
                // ========================================
                // STEP 4.1: Create ALL Directories
                // ========================================
                $updateLog[] = ti('📁 Erstelle Verzeichnisse...', '📁 Creating directories...');
                
                $directories = [
                    __DIR__ . '/media', __DIR__ . '/media/pages', __DIR__ . '/media/blog',
                    __DIR__ . '/media/uploads', __DIR__ . '/media/logo', __DIR__ . '/media/favicon',
                    __DIR__ . '/cache', __DIR__ . '/cache/assets', __DIR__ . '/cache/queries',
                    __DIR__ . '/.opcache', __DIR__ . '/logs',
                    __DIR__ . '/customizations', __DIR__ . '/customizations/features',
                    __DIR__ . '/customizations/admin-pages', __DIR__ . '/customizations/config',
                    __DIR__ . '/backups'
                ];
                
                foreach ($directories as $dir) {
                    if (!is_dir($dir)) {
                        mkdir($dir, 0755, true);
                        $updateLog[] = '✅ ' . str_replace(__DIR__ . '/', '', $dir);
                    }
                }
                
                // ========================================
                // STEP 4.2: Create .env file
                // ========================================
                $updateLog[] = ti('📝 Erstelle .env Datei...', '📝 Creating .env file...');
                
                $envContent = "; ManiCMS Configuration\n\n";
                $envContent .= "DB_HOST={$db['host']}\n";
                $envContent .= "DB_NAME={$db['name']}\n";
                $envContent .= "DB_USER={$db['user']}\n";
                $envContent .= "DB_PASS={$db['pass']}\n\n";
                
                $envContent .= "APP_URL={$site['url']}\n";
                $envContent .= "APP_ENV=production\n";
                $envContent .= "APP_DEBUG=false\n";
                $envContent .= "APP_LANGUAGE={$installLang}\n";
                $envContent .= "FORCE_SSL=" . ($site['force_ssl'] ? 'true' : 'false') . "\n\n";
                
                $envContent .= "SESSION_NAME=CMS_SESSION\n";
                $envContent .= "SESSION_LIFETIME=7200\n\n";
                
                // License section
                if (!empty($license['key'])) {
                    $envContent .= "LICENSE_KEY={$license['key']}\n";
                    $envContent .= "LICENSE_VALID=" . ($license['valid'] ? 'true' : 'false') . "\n";
                    if ($license['valid'] && !empty($license['data']['activated_at'])) {
                        $envContent .= "LICENSE_ACTIVATED={$license['data']['activated_at']}\n";
                    }
                } else {
                    $envContent .= "; LICENSE_KEY=\n; LICENSE_VALID=false\n";
                }
                
                if (!file_put_contents(__DIR__ . '/.env', $envContent)) {
                    throw new Exception(ti('.env konnte nicht erstellt werden', 'Could not create .env file'));
                }
                
                @chmod(__DIR__ . '/.env', 0600);
                $updateLog[] = ti('✅ .env Datei erstellt', '✅ .env file created');
                
                // ========================================
                // STEP 4.3: Install Database
                // ========================================
                $updateLog[] = ti('🗄️ Installiere Datenbank...', '🗄️ Installing database...');
                
                $dsn = "mysql:host={$db['host']};dbname={$db['name']};charset=utf8mb4";
                $pdo = new PDO($dsn, $db['user'], $db['pass'], [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
                ]);
                
                require_once __DIR__ . '/database_installer.php';
                $installer = new DatabaseInstaller($pdo);
                $installResult = $installer->install();
                
                if (!$installResult['success']) {
                    throw new Exception('Database installation failed');
                }
                
                $updateLog = array_merge($updateLog, $installResult['log']);
                
                // ========================================
                // STEP 4.4: Create Admin & Homepage
                // ========================================
                require_once __DIR__ . '/cms-core/backend/core/init.php';
                
                $hashedPassword = password_hash($site['pass'], PASSWORD_DEFAULT);
                db()->execute(
                    "INSERT INTO users (username, email, password, password_hash, role, created_at) VALUES (?, ?, ?, ?, 'admin', NOW())",
                    [$site['user'], $site['email'], $hashedPassword, $hashedPassword]
                );
                
                db()->execute(
                    "INSERT INTO pages (title, slug, status, content, created_at, last_modified) VALUES (?, 'index', 'published', '[]', NOW(), NOW())",
                    [ti('Startseite', 'Home')]
                );
                
                db()->execute("UPDATE settings SET setting_value = ? WHERE setting_key = 'site_name'", [$site['name']]);
                db()->execute("UPDATE settings SET setting_value = ? WHERE setting_key = 'site_email'", [$site['email']]);
                
                $updateLog[] = ti('✅ Admin & Homepage erstellt', '✅ Admin & Homepage created');
                
                // License info
                if (!empty($license['key'])) {
                    if ($license['valid']) {
                        $updateLog[] = ti('🎫 Lizenz aktiviert - Updates verfügbar', '🎫 License activated - updates available');
                    } else {
                        $updateLog[] = ti('⚠️ Lizenz nicht validiert - später prüfen', '⚠️ License not validated - check later');
                    }
                } else {
                    $updateLog[] = ti('ℹ️ Ohne Lizenz - CMS funktioniert, Updates manuell', 'ℹ️ Without license - CMS works, updates manual');
                }
                
                $updateLog[] = ti('✅ Installation abgeschlossen!', '✅ Installation complete!');
                
                unset($_SESSION['install_db'], $_SESSION['install_site'], $_SESSION['install_license']);
                $success = true;
                
            } catch (Exception $e) {
                $errors[] = ti('Installation fehlgeschlagen: ', 'Installation failed: ') . $e->getMessage();
            }
        }
    }
}

$licenseData = $_SESSION['install_license'] ?? [];
$dbData = $_SESSION['install_db'] ?? [];
$siteData = $_SESSION['install_site'] ?? [];
?>
<!DOCTYPE html>
<html lang="<?= $installLang ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= ti('ManiCMS Installation', 'ManiCMS Installation') ?></title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .container {
            background: white;
            border-radius: 16px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            max-width: 600px;
            width: 100%;
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px;
            text-align: center;
        }
        .header h1 { font-size: 32px; margin-bottom: 8px; }
        .progress-bar {
            display: flex;
            background: rgba(255,255,255,0.2);
            margin-top: 24px;
            border-radius: 8px;
            overflow: hidden;
        }
        .progress-step {
            flex: 1;
            padding: 12px 8px;
            text-align: center;
            font-size: 12px;
            font-weight: 600;
            background: rgba(255,255,255,0.1);
        }
        .progress-step.active {
            background: rgba(255,255,255,0.3);
        }
        .body { padding: 40px; }
        .form-group { margin-bottom: 24px; }
        .form-label { display: block; font-weight: 600; color: #374151; margin-bottom: 8px; }
        .form-input { width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 16px; }
        .form-input:focus { outline: none; border-color: #667eea; }
        .form-hint { font-size: 13px; color: #6b7280; margin-top: 6px; }
        .checkbox-label { display: flex; align-items: center; gap: 12px; padding: 16px; background: #f9fafb; border: 2px solid #e5e7eb; border-radius: 8px; cursor: pointer; }
        .btn { width: 100%; padding: 14px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 8px; font-size: 16px; font-weight: 600; cursor: pointer; transition: transform 0.2s; }
        .btn:hover { transform: translateY(-2px); }
        .btn-secondary { background: #6b7280; }
        .alert-error { background: #fee2e2; border: 2px solid #ef4444; color: #991b1b; padding: 16px; border-radius: 8px; margin-bottom: 24px; }
        .language-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-top: 32px; }
        .language-card { padding: 32px 24px; border: 3px solid #e5e7eb; border-radius: 12px; text-align: center; cursor: pointer; transition: all 0.3s; }
        .language-card:hover { border-color: #667eea; transform: translateY(-4px); }
        .language-icon { font-size: 64px; margin-bottom: 16px; }
        .language-name { font-size: 20px; font-weight: 600; }
        .log { background: #f9fafb; padding: 20px; border-radius: 8px; margin: 24px 0; max-height: 400px; overflow-y: auto; font-family: monospace; font-size: 13px; line-height: 1.6; }
        .success-icon { font-size: 80px; text-align: center; margin-bottom: 24px; }
        .license-box { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 32px; border-radius: 12px; text-align: center; margin: 24px 0; }
        .license-box h3 { font-size: 24px; margin-bottom: 16px; }
        .license-box p { margin-bottom: 24px; opacity: 0.9; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🚀 ManiCMS</h1>
            <p><?= ti('Installation', 'Installation') ?></p>
            
            <?php if ($step > 0 && $step < 4): ?>
            <div class="progress-bar">
                <div class="progress-step <?= $step >= 1 ? 'active' : '' ?>">1. <?= ti('Lizenz', 'License') ?></div>
                <div class="progress-step <?= $step >= 2 ? 'active' : '' ?>">2. <?= ti('DB', 'DB') ?></div>
                <div class="progress-step <?= $step >= 3 ? 'active' : '' ?>">3. <?= ti('Website', 'Website') ?></div>
                <div class="progress-step <?= $step >= 4 ? 'active' : '' ?>">4. <?= ti('Fertig', 'Done') ?></div>
            </div>
            <?php endif; ?>
        </div>
        
        <div class="body">
            <?php if (!empty($errors)): ?>
                <div class="alert-error">
                    <strong>❌ <?= ti('Fehler:', 'Errors:') ?></strong>
                    <ul style="margin-left: 20px; margin-top: 8px;">
                        <?php foreach ($errors as $error): ?>
                            <li><?= htmlspecialchars($error) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <?php if ($step === 0): ?>
                <h2 style="text-align: center; margin-bottom: 32px;">
                    <?= ti('Sprache wählen', 'Choose Language') ?>
                </h2>
                <form method="POST">
                    <div class="language-grid">
                        <div class="language-card" onclick="this.querySelector('input').checked = true; this.closest('form').submit();">
                            <input type="radio" name="language" value="de" style="display:none;" <?= $installLang === 'de' ? 'checked' : '' ?>>
                            <div class="language-icon">🇩🇪</div>
                            <div class="language-name">Deutsch</div>
                        </div>
                        
                        <div class="language-card" onclick="this.querySelector('input').checked = true; this.closest('form').submit();">
                            <input type="radio" name="language" value="en" style="display:none;" <?= $installLang === 'en' ? 'checked' : '' ?>>
                            <div class="language-icon">🇬🇧</div>
                            <div class="language-name">English</div>
                        </div>
                    </div>
                </form>
                
            <?php elseif ($step === 1): ?>
                <div class="license-box">
                    <h3>🎫 <?= ti('Lizenz-Key (Optional)', 'License Key (Optional)') ?></h3>
                    <p><?= ti('Mit einer Lizenz erhältst du automatische Updates. Ohne Lizenz funktioniert das CMS vollständig.', 'With a license you get automatic updates. Without a license the CMS works fully.') ?></p>
                </div>
                <form method="POST">
                    <div class="form-group">
                        <label class="form-label"><?= ti('Lizenz-Key', 'License Key') ?></label>
                        <input type="text" 
                               name="license_key" 
                               class="form-input" 
                               placeholder="MANI-XXXX-XXXX-XXXX-XXXX"
                               style="font-family: monospace; letter-spacing: 1px;"
                               value="<?= htmlspecialchars($licenseData['key'] ?? '') ?>">
                        <div class="form-hint">
                            <?= ti('Hast du noch keine Lizenz?', 'Don\'t have a license yet?') ?> 
                            <a href="https://manidesign.de" target="_blank" style="color: #667eea;">
                                <?= ti('Jetzt kaufen', 'Buy now') ?>
                            </a>
                        </div>
                    </div>
                    <button type="submit" class="btn"><?= ti('Weiter', 'Next') ?> →</button>
                    <button type="submit" name="skip_license" value="1" class="btn btn-secondary" style="margin-top: 12px;">
                        <?= ti('Ohne Lizenz fortfahren', 'Continue without license') ?>
                    </button>
                </form>
                
            <?php elseif ($step === 2): ?>
                <h2><?= ti('Datenbank-Konfiguration', 'Database Configuration') ?></h2>
                <form method="POST" style="margin-top: 24px;">
                    <div class="form-group">
                        <label class="form-label"><?= ti('Host', 'Host') ?></label>
                        <input type="text" name="db_host" class="form-input" value="<?= htmlspecialchars($dbData['host'] ?? 'localhost') ?>" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label"><?= ti('Datenbank-Name', 'Database Name') ?></label>
                        <input type="text" name="db_name" class="form-input" value="<?= htmlspecialchars($dbData['name'] ?? '') ?>" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label"><?= ti('Benutzer', 'User') ?></label>
                        <input type="text" name="db_user" class="form-input" value="<?= htmlspecialchars($dbData['user'] ?? '') ?>" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label"><?= ti('Passwort', 'Password') ?></label>
                        <input type="password" name="db_pass" class="form-input">
                    </div>
                    <button type="submit" class="btn"><?= ti('Weiter', 'Next') ?> →</button>
                </form>
                
            <?php elseif ($step === 3): ?>
                <h2><?= ti('Website-Konfiguration', 'Website Configuration') ?></h2>
                <form method="POST" style="margin-top: 24px;">
                    <div class="form-group">
                        <label class="form-label"><?= ti('Website-URL', 'Website URL') ?></label>
                        <input type="text" name="site_url" class="form-input" placeholder="https://example.com" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label"><?= ti('Website-Name', 'Website Name') ?></label>
                        <input type="text" name="site_name" class="form-input" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label"><?= ti('Admin-Email', 'Admin Email') ?></label>
                        <input type="email" name="admin_email" class="form-input" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label"><?= ti('Admin-Benutzername', 'Admin Username') ?></label>
                        <input type="text" name="admin_user" class="form-input" value="admin" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label"><?= ti('Admin-Passwort', 'Admin Password') ?></label>
                        <input type="password" name="admin_pass" class="form-input" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label"><?= ti('Passwort bestätigen', 'Confirm Password') ?></label>
                        <input type="password" name="admin_pass_confirm" class="form-input" required>
                    </div>
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input type="checkbox" name="force_ssl" value="1" checked>
                            <span><?= ti('SSL erzwingen (HTTPS)', 'Force SSL (HTTPS)') ?></span>
                        </label>
                    </div>
                    <button type="submit" class="btn"><?= ti('Installieren', 'Install') ?> →</button>
                </form>
                
            <?php elseif ($step === 4 && $success): ?>
                <div class="success-icon">✅</div>
                <h2 style="text-align: center;"><?= ti('Installation erfolgreich!', 'Installation Successful!') ?></h2>
                
                <?php if (!empty($updateLog)): ?>
                <div class="log">
                    <?php foreach ($updateLog as $log): ?>
                        <div><?= htmlspecialchars($log) ?></div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <div style="text-align: center; margin-top: 32px;">
                    <a href="/admin" class="btn" style="display: inline-block; text-decoration: none;">
                        <?= ti('Zum Admin-Bereich', 'Go to Admin Area') ?> →
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>